package ru.yandex.chemodan.videostreaming.framework.hls;

import java.io.InputStream;
import java.util.function.Consumer;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.collection.impl.AbstractPrefetchingIterator;
import ru.yandex.chemodan.videostreaming.framework.hls.segmentprovider.HlsSegmentSource;
import ru.yandex.chemodan.videostreaming.framework.hls.segmentprovider.HlsStreamingManager;
import ru.yandex.chemodan.videostreaming.framework.hls.segmentprovider.caching.HlsSegmentCache;
import ru.yandex.chemodan.videostreaming.framework.hls.segmentprovider.caching.SegmentIndexes;
import ru.yandex.chemodan.videostreaming.framework.hls.stats.HlsRequestStats;
import ru.yandex.chemodan.videostreaming.framework.hls.videoinfo.StreamingResourceProvider;
import ru.yandex.chemodan.videostreaming.framework.hls.videoinfo.caching.VideoInfoCache;
import ru.yandex.chemodan.videostreaming.framework.m3u.ExtM3UMasterPlaylist;
import ru.yandex.chemodan.videostreaming.framework.m3u.ExtM3UMediaPlaylist;
import ru.yandex.chemodan.videostreaming.framework.m3u.ExtM3UPlaylists;
import ru.yandex.chemodan.videostreaming.framework.media.units.MediaTime;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.monica.annotation.GroupByDefault;
import ru.yandex.misc.monica.annotation.MonicaContainer;
import ru.yandex.misc.monica.annotation.MonicaMetric;
import ru.yandex.misc.monica.core.blocks.Meter;
import ru.yandex.misc.monica.core.blocks.MeterMap;
import ru.yandex.misc.monica.core.blocks.Statistic;
import ru.yandex.misc.monica.core.name.MetricGroupName;
import ru.yandex.misc.monica.core.name.MetricName;
import ru.yandex.misc.time.Stopwatch;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class HlsManager<T> implements MonicaContainer {
    @SuppressWarnings("unused")
    private static final Logger logger = LoggerFactory.getLogger(HlsManager.class);

    public static final String MASTER_PLAYLIST_FILENAME = "master-playlist.m3u8";

    public static final String PLAYLIST_FILENAME = "playlist.m3u8";

    public static final String SEGMENT_MEDIA_TYPE = "video/mp2t";

    public static final String SEGMENT_CACHE_MISSING_PARAM = "ab";

    private static final DynamicProperty<Integer> maxHeightCapping =
            DynamicProperty.cons("streaming-max-height-capping", 0);

    private final StreamingResourceProvider<T> resourceProvider;

    private final HlsUrlBuilder<T> urlBuilder;

    private final HlsStreamingManager<T> streamingManager;

    private final StreamingParams streamingParams;

    private final HlsSegmentCache<T> segmentCache;

    private final VideoInfoCache<T> videoInfoCache;

    @MonicaMetric
    @GroupByDefault
    private final Statistic segmentTranscodingDurations = new Statistic();

    @GroupByDefault
    @MonicaMetric
    private final MeterMap segmentResponseRps = new MeterMap();

    @GroupByDefault
    @MonicaMetric
    private final Meter errorRps = new Meter();

    public HlsManager(
            HlsStreamingManager<T> streamingManager,
            StreamingResourceProvider<T> resourceProvider,
            HlsParams hlsParams,
            HlsUrlBuilder<T> urlBuilder,
            HlsSegmentCache<T> segmentCache,
            VideoInfoCache<T> videoInfoCache)
    {
        this.urlBuilder = urlBuilder;
        this.streamingManager = streamingManager;
        this.resourceProvider = resourceProvider;
        this.streamingParams = hlsParams.getStreamingParams();
        this.segmentCache = segmentCache;
        this.videoInfoCache = videoInfoCache;
    }

    @Override
    public MetricGroupName groupName(String instanceName) {
        return new MetricGroupName("hls", new MetricName("hls"), "HLS metrics");
    }

    public ExtM3UMasterPlaylist buildMasterPlaylist(T sourceMeta, boolean extM3uSessionDataEnabled) {
        return getHlsResource(sourceMeta, true)
                .toMasterPlaylist(
                        getPlaylistItemUri(PLAYLIST_FILENAME),
                        Option.of(maxHeightCapping.get()).filter(maxHeight -> extM3uSessionDataEnabled && maxHeight > 0)
                );
    }

    public ExtM3UMediaPlaylist buildPlaylist(T sourceMeta, HlsStreamQuality quality) {
        HlsResource hlsResource = getHlsResource(sourceMeta, true);
        hlsResource.checkIfQualityExists(quality);

        if (streamingParams.isDeleteCache()) {
            try {
                segmentCache.clearCacheForSource(sourceMeta);
                videoInfoCache.clearCacheForSource(sourceMeta);
            } catch (UnsupportedOperationException ex) {
                logger.warn("Clear cache operation not implemented");
            }
        }
        return consPlaylist(sourceMeta, quality, getSegmentation(hlsResource));
    }

    private MediaSegmentation getSegmentation(HlsResource hlsResource) {
        return MediaSegmentation.fromSegmentAndTotalDurations(
                streamingParams.getSegmentDuration(),
                hlsResource.getTotalDuration()
        );
    }

    public SegmentResponseHandler consResponseHandler(T sourceMeta, HlsStreamQuality quality, int segmentIndex) {
        return new SegmentResponseHandler(sourceMeta, quality, segmentIndex);
    }

    private HlsResource getHlsResource(T sourceMeta, boolean forPlaylist) {
        return MasterSlaveContextHolder.withPolicy(forPlaylist ? MasterSlavePolicy.R_MS : MasterSlavePolicy.R_SM,
                () -> resourceProvider.getResource(sourceMeta)
        );
    }

    public class SegmentResponseHandler {
        final HlsSegmentMeta<T> segmentMeta;

        final Stopwatch stopwatch = Stopwatch.createAndStart();

        SegmentResponseHandler(T sourceMeta, HlsStreamQuality quality, int segmentIndex) {
            HlsResource.Stream stream = getHlsResource(sourceMeta, false)
                    .getStream(quality);
            this.segmentMeta = new HlsSegmentMeta<>(
                    stream,
                    new HlsSegmentRegion(segmentIndex, streamingParams.getSegmentDuration(), stream),
                    sourceMeta
            );
        }

        public void prefetch() {
            exec(() -> {
                streamingManager.prefetch(segmentMeta);
                logger.info("Prefetched segment#{}, took {}", segmentMeta.getIndex(), stopwatch.duration());
            });
        }

        public void stream(Consumer<InputStream> consumer) {
            exec(() -> {
                HlsSegmentSource src = streamingManager.stream(segmentMeta, consumer);
                HlsRequestStats.getCurrent().setSource(src);
                Duration duration = stopwatch.duration();
                logger.info("Received segment#{} from {}, took {}", segmentMeta.getIndex(), src, duration);

                if (src == HlsSegmentSource.FFMPEG) {
                    segmentTranscodingDurations.update(duration.getMillis());
                }
                segmentResponseRps.inc(src.metricName);

            });
        }

        public void streamLocalPrefetch(Consumer<InputStream> consumer) {
            exec(() -> streamingManager.streamLocalPrefetch(segmentMeta, consumer));
        }

        void exec(Runnable runnable) {
            logDuration();
            try {
                runnable.run();
            } catch (RuntimeException ex) {
                errorRps.inc();
                throw ex;
            }
        }

        void logDuration() {
            HlsRequestStats.getCurrent().setDuration(segmentMeta.getActualDuration());
        }
    }

    private ExtM3UMediaPlaylist consPlaylist(T sourceMeta, HlsStreamQuality quality, MediaSegmentation segmentation) {
        SegmentIndexes cachedSegmentIndexes =
                segmentCache.getCachedSegmentIndexes(sourceMeta, quality, streamingParams.getSegmentDuration());
        String segmentExtension = getSegmentExtension(quality);
        return ExtM3UPlaylists.mediaVodAllowCacheEndList(
                new AbstractPrefetchingIterator<Tuple2<String, Number>>() {
                    int i = 0;

                    @Override
                    protected Option<Tuple2<String, Number>> fetchNext() {
                        return consUriAndDurationO(i++);
                    }

                    private Option<Tuple2<String, Number>> consUriAndDurationO(int n) {
                        return Option.when(n < segmentation.getSegmentCount(), () -> consUriAndDuration(n));
                    }

                    private Tuple2<String, Number> consUriAndDuration(int n) {
                        String uri =getPlaylistItemUri(
                                n + "." + segmentExtension,
                                getSegmentExtraParams(n, cachedSegmentIndexes));
                        return new Tuple2<>(uri, getPlaylistSegmentDuration(segmentation.getActualSegmentDuration(n)));
                    }
                }
        );
    }

    private Number getPlaylistSegmentDuration(MediaTime duration) {
        // do not convert to ternary operator:
        //   * then using ternary both branches return double
        //   * then using "if" second branch returns int
        if (streamingParams.isUseFloatSegmentDurations()) {
            return DurationUtil.toSecondsDouble(duration);
        } else {
            return DurationUtil.toSecondsCeil(duration);
        }
    }

    private static String getSegmentExtension(HlsStreamQuality quality) {
        return quality == HlsStreamQuality._AAC ? "aac" : "ts";
    }

    private static Tuple2List<String, String> getSegmentExtraParams(int segmentIndex,
            SegmentIndexes cachedSegmentIndexes)
    {
        return Tuple2List.fromPairs(SEGMENT_CACHE_MISSING_PARAM,
                !cachedSegmentIndexes.contains(segmentIndex) ? "1" : "0");
    }

    private String getPlaylistItemUri(String filename) {
        return getPlaylistItemUri(filename, Tuple2List.tuple2List());
    }

    private String getPlaylistItemUri(String filename, Tuple2List<String, String> extraParams) {
        return urlBuilder.getRelativePlaylistItemUri(filename, extraParams);
    }
}
