package ru.yandex.chemodan.videostreaming.framework.hls;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.chemodan.videostreaming.framework.m3u.ExtM3UMasterPlaylist;
import ru.yandex.chemodan.videostreaming.framework.media.MediaInfo;
import ru.yandex.chemodan.videostreaming.framework.media.units.BitRate;
import ru.yandex.chemodan.videostreaming.framework.media.units.FrameRate;
import ru.yandex.chemodan.videostreaming.framework.media.units.MediaTime;
import ru.yandex.chemodan.videostreaming.framework.web.HlsErrorSource;
import ru.yandex.misc.image.Dimension;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class HlsResource extends DefaultObject {
    private static final BitRate ZERO_BIT_RATE = new BitRate(0);

    private static final Dimension ZERO_DIMENSION = new Dimension(0, 0);

    private static final BitRate MAX_AUDIO_BIT_RATE = BitRate.fromKbps(128);

    private static final HlsStreamQuality SEPARATE_AUDIO_FORMAT = HlsStreamQuality._AAC;

    private static final String MAX_HEIGHT_CAPPING_DATA_ID = "com.yandex.video.capping.maxheight";

    public final MediaInfo fileInformation;

    public final boolean separateAudio;

    public final transient MapF<HlsStreamQuality, Stream> qualityToStream;

    public HlsResource(MediaInfo fileInformation) {
        this(fileInformation, false);
    }

    public HlsResource(MediaInfo fileInformation, boolean separateAudio){
            this(fileInformation, getStreamQualities(fileInformation), separateAudio);
        }

    private HlsResource(MediaInfo fileInformation, ListF<HlsStreamQuality> qualities, boolean separateAudio) {
        this.fileInformation = fileInformation;
        this.qualityToStream = Cf.x(
                qualities.plus(Option.when(separateAudio, SEPARATE_AUDIO_FORMAT))
                        .zipWith(Stream::new)
                        .toJavaLinkedHashMap()
        ).unmodifiable();
        this.separateAudio = separateAudio;
    }

    private static ListF<HlsStreamQuality> getStreamQualities(MediaInfo fileInformation) {
        return HlsStreamQuality.getSuitableFor(
                fileInformation.getVideoStreamO()
                        .filterMap(MediaInfo.VideoStream::getDimension),
                fileInformation.hasAudio()
        );
    }

    public HlsResource filterQualities(Function1B<HlsStreamQuality> filterF) {
        return new HlsResource(fileInformation, getStreamQualities().filter(filterF), separateAudio);
    }

    ListF<HlsStreamQuality> getStreamQualities() {
        return qualityToStream.keys();
    }

    public MediaInfo.VideoStream getVideoInfo() {
        return fileInformation.getVideoStream();
    }

    public Option<BitRate> getVideoBitrateO() {
        return fileInformation.getVideoBitRateO();
    }

    public MediaTime getTotalDuration() {
        return getTotalDurationO().get();
    }

    public Option<MediaTime> getTotalDurationO() {
        return fileInformation.getDuration();
    }

    public Stream getStream(HlsStreamQuality quality) {
        return qualityToStream.getO(quality)
                .getOrThrow(() -> new UnsupportedQualityException(quality));
    }

    public void checkIfQualityExists(HlsStreamQuality quality) {
        if (!qualityToStream.containsKeyTs(quality)) {
            throw new UnsupportedQualityException(quality);
        }
    }

    public CollectionF<Stream> getStreams() {
        return qualityToStream.values();
    }

    public ExtM3UMasterPlaylist toMasterPlaylist(String playlistUri, Option<Integer> maxHeightCappingO) {
        return new ExtM3UMasterPlaylist(
                Cf.<ExtM3UMasterPlaylist.Item>list()
                        .plus(maxHeightCappingO.map(this::buildMaxHeightCappingExtM3USessionData))
                        .plus(getPlaylistStreams(playlistUri, Option.when(separateAudio, "aac")))
                        .plus(Option.when(separateAudio, () -> consPlaylistAudio(playlistUri)))
        );
    }

    // https://st.yandex-team.ru/PLAYERWEB-1769
    private ExtM3UMasterPlaylist.SessionDataItem buildMaxHeightCappingExtM3USessionData(Integer maxHeight) {
        return new ExtM3UMasterPlaylist.SessionDataItem(MAX_HEIGHT_CAPPING_DATA_ID, maxHeight.toString());
    }

    private ListF<ExtM3UMasterPlaylist.Stream> getPlaylistStreams(String playlistUri, Option<String> audioName) {
        return getStreams()
                .filterNot(stream -> HlsStreamQuality._AAC.equals(stream.getQuality()))
                .sortedByDesc(Stream::getQuality)
                .map(stream -> stream.toMasterPlaylistItem(playlistUri, audioName));
    }

    private ExtM3UMasterPlaylist.Media consPlaylistAudio(String playlistUri) {
        return new ExtM3UMasterPlaylist.Media("aac", "default", true, true,
                SEPARATE_AUDIO_FORMAT.toRequestParamValue() + "/" + playlistUri
        );
    }

    public Option<Dimension> getRotatedDimensionO() {
        return fileInformation.getRotatedDimensionO();
    }

    public HlsStreamQuality getBestQuality() {
        return getStreamQualities().max(HlsStreamQuality.DIMENSION_COMPARATOR);
    }

    public class Stream extends DefaultObject {
        private final HlsStreamQuality quality;

        public Stream(HlsStreamQuality quality) {
            this.quality = quality;
        }

        public HlsStreamQuality getQuality() {
            return quality;
        }

        public Dimension getDimension() {
            return fileInformation.hasVideo()
                    ? quality.getDimensionFor(getVideoInfo())
                    : ZERO_DIMENSION;
        }

        public BitRate getVideoBitrate() {
            if (!fileInformation.hasVideo()) {
                return ZERO_BIT_RATE;
            }

            return BitrateUtils.videoBitRateForDimension(getDimension());
        }

        public Option<BitRate> getAudioBitrateO() {
            return fileInformation.getAudioStreamO()
                    .map(format -> format.getBitRate().getOrElse(MAX_AUDIO_BIT_RATE))
                    .map(bitRate -> min(bitRate, MAX_AUDIO_BIT_RATE));
        }

        private BitRate getAudioBitRate() {
            return getAudioBitrateO()
                    .getOrElse(ZERO_BIT_RATE);
        }

        public BitRate getTotalBitrate() {
            return new BitRate(getVideoBitrate().getBs() + getAudioBitRate().getBs());
        }

        public MediaInfo getFileInformation() {
            return fileInformation;
        }

        public MediaTime getTotalDuration() {
            return HlsResource.this.getTotalDuration();
        }

        private ExtM3UMasterPlaylist.Stream toMasterPlaylistItem(String playlistUri, Option<String> audioNameO) {
            String fullUri = StringUtils.join(
                    Cf.list(getQuality().toRequestParamValue(), playlistUri),
                    "/"
            );
            return new ExtM3UMasterPlaylist.Stream(getTotalBitrate(), getDimension(), fullUri, audioNameO);
        }

        public Option<FrameRate> getFrameRateO() {
            return fileInformation.getFrameRateO();
        }
    }

    private static BitRate min(BitRate bitRate1, BitRate bitRate2) {
        return bitRate1.getBs() <= bitRate2.getBs() ? bitRate1 : bitRate2;
    }

    private static class UnsupportedQualityException extends RuntimeException implements HlsErrorSource {
        private final HlsStreamQuality quality;

        public UnsupportedQualityException(HlsStreamQuality quality) {
            this.quality = quality;
        }

        @Override
        public HlsError getHlsError() {
            return new HlsError(HttpStatus.SC_404_NOT_FOUND, "Quality " + quality + " is not supported");
        }
    }
}
