package ru.yandex.chemodan.videostreaming.framework.hls;

import java.util.Comparator;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.videostreaming.framework.media.MediaInfo;
import ru.yandex.chemodan.videostreaming.framework.media.units.Fraction;
import ru.yandex.misc.enums.EnumResolver;
import ru.yandex.misc.image.Dimension;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author Lev Tolmachev
 */
public enum HlsStreamQuality {
    // This isn't really 360p
    _DEFAULT(new Dimension(408, 204)),

    _AUDIO(new Dimension(0, 0)),

    _AAC(new Dimension(0, 0)),

    _240P(new Dimension(426, 240)),
    _360P(new Dimension(640, 360)),
    _480P(new Dimension(852, 480)),
    _720P(new Dimension(1280, 720)),
    _1080P(new Dimension(1920, 1080))
    ;

    public static final Comparator<HlsStreamQuality> DIMENSION_COMPARATOR =
            Comparator.comparingLong(q -> q.getDimension().getSquare());

    private static final SetF<HlsStreamQuality> SPECIAL_QUALITIES = Cf.set(_DEFAULT, _AUDIO, _AAC);

    public static final ListF<HlsStreamQuality> MASTER_QUALITIES =
            Cf.list(values())
                    .filterNot(SPECIAL_QUALITIES::containsTs);

    static final ListF<HlsStreamQuality> AUDIO_QUALITIES = Cf.list(_AUDIO);

    public final Dimension dimension;

    HlsStreamQuality(Dimension dimension) {
        this.dimension = dimension;
    }

    public Dimension getDimension() {
        return dimension;
    }

    public static HlsStreamQuality parse(String dimensionStr) {
        return parseO(dimensionStr)
                .getOrThrow(() -> new IllegalArgumentException("Unknown quality = " + dimensionStr));
    }

    public static Option<HlsStreamQuality> parseO(String dimensionStr) {
        return dimensionStr.startsWith("-")
                ? R.valueOfO(dimensionStr)
                : R.valueOfO("-" + dimensionStr);
    }

    public Dimension getDimensionFor(MediaInfo.VideoStream videoStream) {
        return getDimensionFor(
                videoStream.getDimension().get(),
                videoStream.getOrCalculateDisplayAspectRatio().get(),
                videoStream.getRotation()
                        .isMatch(rotation -> rotation.getCcwDegrees() % 180 != 0)
        );
    }

    private Dimension getDimensionFor(Dimension srcDimension, Fraction aspectRatio, boolean isRotated) {
        if (isRotated) {
            aspectRatio = aspectRatio.invert();
            srcDimension = srcDimension.rotate();
        }

        int tgtHeight = dimension.getHeight();
        if (srcDimension.getHeight() <= tgtHeight) {
            return srcDimension.makeSizesEven();
        } else {
            int width = (int) Math.round(aspectRatio.doubleValue() * tgtHeight);
            return Dimension.valueOf(width, tgtHeight)
                    .makeSizesEven();
        }
    }

    public String toRequestParamValue() {
        return StringUtils.removeStart(name().toLowerCase(), "_");
    }

    public int getHeight() {
        return dimension.getHeight();
    }

    public static ListF<HlsStreamQuality> getSuitableFor(Option<Dimension> dimensionO, boolean hasAudio) {
        return dimensionO.map(HlsStreamQuality::getSuitableFor)
                .orElse(Option.when(hasAudio, AUDIO_QUALITIES))
                .getOrElse(Cf::list);
    }

    public static ListF<HlsStreamQuality> getSuitableFor(Dimension dimension) {
        int origHeight = dimension.getHeight();
        HlsStreamQuality maxQuality = MASTER_QUALITIES.filter(q -> q.getHeight() >= origHeight)
                .minO()
                .getOrElse(MASTER_QUALITIES::max);
        return MASTER_QUALITIES.filter(q -> q.getHeight() <= maxQuality.getHeight());
    }

    private static final EnumResolver<HlsStreamQuality> R = EnumResolver.er(HlsStreamQuality.class);
}
