package ru.yandex.chemodan.videostreaming.framework.hls;

import lombok.Getter;
import lombok.Value;
import org.joda.time.Duration;

import ru.yandex.chemodan.videostreaming.framework.media.units.MediaTime;
import ru.yandex.commune.dynproperties.DynamicProperty;

/**
 * @author Dmitriy Amelin (lemeh)
 */
@Value
public class MediaSegmentation {
    private static final MediaTime VIDEO_DURATION_LIMIT = MediaTime.fromDuration(Duration.standardDays(1));

    static final DynamicProperty<Long> lastSegmentMinDuration =
            DynamicProperty.cons("streaming-last-segment-min-duration", 120L);

    MediaTime segmentDuration;

    MediaTime totalDuration;

    @Getter(lazy = true)
    int segmentCount = (int) calcSegmentCount();

    public static MediaSegmentation fromSegmentAndTotalDurations(MediaTime segmentDuration, MediaTime totalDuration) {
        // https://st.yandex-team.ru/CHEMODAN-37228
        if (totalDuration.isLongerThan(VIDEO_DURATION_LIMIT)) {
            totalDuration = VIDEO_DURATION_LIMIT;
        }

        return new MediaSegmentation(segmentDuration, totalDuration);
    }

    public MediaTime getActualSegmentDuration(int index) {
        return isLastSegment(index)
                ? totalDuration.minus(segmentDuration.multipliedBy(index))
                : segmentDuration;
    }

    public boolean isLastSegment(int index) {
        return getSegmentCount() == index + 1;
    }

    private long calcSegmentCount() {
        return calcUncutSegmentCount() + (addExtraSegment() ? 1 : 0);
    }

    private long calcUncutSegmentCount() {
        return totalDuration.getMicros() / segmentDuration.getMicros();
    }

    private boolean addExtraSegment() {
        return totalDuration.gt(MediaTime.ZERO) && (totalDuration.lt(segmentDuration) || remainderTimeIsLongEnough());
    }

    // https://st.yandex-team.ru/CHEMODAN-57762
    private boolean remainderTimeIsLongEnough() {
        return new MediaTime(totalDuration.getMicros() % segmentDuration.getMicros())
                .isLongerThan(getLastSegmentMinDuration());
    }

    private static MediaTime getLastSegmentMinDuration() {
        return MediaTime.millis(lastSegmentMinDuration.get());
    }
}
