package ru.yandex.chemodan.videostreaming.framework.hls;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.videostreaming.framework.media.MediaInfo;
import ru.yandex.chemodan.videostreaming.framework.media.units.BitRate;
import ru.yandex.chemodan.videostreaming.framework.media.units.FrameRate;
import ru.yandex.chemodan.videostreaming.framework.media.units.MediaTime;
import ru.yandex.chemodan.videostreaming.framework.media.units.SampleFrequency;
import ru.yandex.chemodan.videostreaming.framework.util.MediaTimeAsMillisMarshaller;
import ru.yandex.chemodan.videostreaming.framework.util.MediaTimeAsMillisUnmarshaller;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.MembersToBind;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.bender.config.BenderConfiguration;
import ru.yandex.misc.bender.config.CustomMarshallerUnmarshallerFactoryBuilder;
import ru.yandex.misc.bender.serialize.BenderSerializer;
import ru.yandex.misc.image.Dimension;
import ru.yandex.misc.lang.CharsetUtils;
import ru.yandex.misc.lang.DefaultObject;

/**
 * @author Dmitriy Amelin (lemeh)
 * @see <a href="https://st.yandex-team.ru/CHEMODAN-36508">CHEMODAN-36508</a>
 */
@BenderBindAllFields
public class ShortFileInformation extends DefaultObject {
    private static final BenderSerializer<ShortFileInformation> serializer = Bender.serializer(ShortFileInformation.class,
            BenderConfiguration.cons(
                    MembersToBind.WITH_ANNOTATIONS, false,
                    CustomMarshallerUnmarshallerFactoryBuilder.cons()
                            .add(MediaTime.class,
                                    new MediaTimeAsMillisMarshaller(),
                                    new MediaTimeAsMillisUnmarshaller())
                            .add(FrameRate.class, FrameRate.marshaller, FrameRate.unmarshaller)
                            .build())
    );

    @BenderPart(name="fmt")
    public final String format;

    @BenderPart(name="len")
    public final Option<MediaTime> duration;

    @BenderPart(name="bps")
    public final Option<BitRate> bitRate;

    @BenderPart(name="video")
    public final Option<Video> video;

    @BenderPart(name="audio")
    public final Option<Audio> audio;

    private ShortFileInformation(String format, Option<MediaTime> duration, Option<BitRate> bitRate,
            Option<Video> video, Option<Audio> audio)
    {
        this.format = format;
        this.duration = duration;
        this.bitRate = bitRate;
        this.video = video;
        this.audio = audio;
    }

    public static ShortFileInformation cons(MediaInfo fileInfo) {
        return new ShortFileInformation(
                fileInfo.getFormat(),
                fileInfo.getDuration(),
                fileInfo.getBitRate(),
                fileInfo.getVideoStreamO().map(Video::cons),
                fileInfo.getAudioStreamO().map(Audio::cons)
        );
    }

    public String toJsonStr() {
        return new String(serializer.serializeJson(this), CharsetUtils.UTF8_CHARSET);
    }

    private static abstract class Stream extends DefaultObject {
        @BenderPart(name="codec")
        public final String codec;

        @BenderPart(name="bps")
        public final Option<BitRate> bitRate;

        protected Stream(String codec, Option<BitRate> bitRate) {
            this.codec = codec;
            this.bitRate = bitRate;
        }
    }

    @BenderBindAllFields
    public static class Video extends Stream {
        @BenderPart(name="size")
        public final Option<String> dimension;

        @BenderPart(name="fps")
        public final Option<FrameRate> frameRate;

        private Video(String codec, Option<BitRate> bitRate, Option<String> dimension, Option<FrameRate> frameRate) {
            super(codec, bitRate);
            this.dimension = dimension;
            this.frameRate = frameRate;
        }

        public static Video cons(MediaInfo.VideoStream videoFormat) {
            return new Video(
                    videoFormat.getCodec(),
                    videoFormat.getBitRate(),
                    videoFormat.getDimension().map(Dimension::toImageMagickSize),
                    videoFormat.getFrameRate()
            );
        }
    }

    @BenderBindAllFields
    public static class Audio extends Stream {
        @BenderPart(name="stereo")
        public final boolean stereo;

        @BenderPart(name="chn")
        public final Option<Integer> channelsCount;

        @BenderPart(name="freq")
        public final Option<SampleFrequency> sampleFrequency;

        private Audio(String codec, Option<BitRate> bitRate, boolean stereo, Option<Integer> channelsCount,
                Option<SampleFrequency> sampleFrequency)
        {
            super(codec, bitRate);
            this.stereo = stereo;
            this.channelsCount = channelsCount;
            this.sampleFrequency = sampleFrequency;
        }

        public static Audio cons(MediaInfo.AudioStream audioFormat) {
            return new Audio(
                    audioFormat.getCodec(),
                    audioFormat.getBitRate(),
                    audioFormat.isStereo(),
                    audioFormat.getChannelsCount(),
                    audioFormat.getSampleFrequency()
            );
        }
    }
}
