package ru.yandex.chemodan.videostreaming.framework.hls;

import java.net.URI;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.videostreaming.framework.util.threadlocal.ReplicableThreadLocal;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class StreamingUrlParams extends DefaultObject {
    private static final ReplicableThreadLocal<StreamingUrlParams> threadLocal = new ReplicableThreadLocal<>();

    private final boolean useHttp;

    private final Option<URI> serverUriO;

    private final Option<String> pathO;

    private final Option<String> queryStringO;

    public StreamingUrlParams() {
        this(false, Option.empty(), Option.empty(), Option.empty());
    }

    private StreamingUrlParams(boolean useHttp, Option<URI> serverUriO, Option<String> pathO,
            Option<String> queryStringO)
    {
        this.useHttp = useHttp;
        this.serverUriO = serverUriO;
        this.pathO = pathO;
        this.queryStringO = queryStringO;
    }

    public static StreamingUrlParams fromHttpServletRequestX(HttpServletRequestX request) {
        return new StreamingUrlParams()
                .withServerUriO(ServerUriUtil.extractServerUriO(request))
                .withPath(request.getRequestURI())
                .withQueryString(request.getQueryString());
    }

    public StreamingUrlParams withUseHttp(boolean useHttp) {
        return new StreamingUrlParams(useHttp, serverUriO, pathO, queryStringO);
    }

    public StreamingUrlParams withServerUriO(Option<URI> serverUriO) {
        return new StreamingUrlParams(useHttp, serverUriO, pathO, queryStringO);
    }

    public StreamingUrlParams withServerUri(URI serverUri) {
        return withServerUriO(Option.of(serverUri));
    }

    public StreamingUrlParams withPathO(Option<String> pathO) {
        return new StreamingUrlParams(useHttp, serverUriO, pathO, queryStringO);
    }

    public StreamingUrlParams withPath(String path) {
        return withPathO(StringUtils.notBlankO(path));
    }

    public StreamingUrlParams withQueryString(String queryString) {
        return new StreamingUrlParams(useHttp, serverUriO, pathO, StringUtils.notBlankO(queryString));
    }

    public StreamingUrlParams withParameters(String... pairs) {
        return withQueryString(
                UrlUtils.listMapToUrlParameters(getTuple2ListFromPairs(pairs))
        );
    }

    public boolean isUseHttp() {
        return useHttp;
    }

    public Option<URI> getServerUriO() {
        return serverUriO;
    }

    public Option<String> getPathO() {
        return pathO;
    }

    public Tuple2List<String, String> getParameters() {
        return queryStringO.map(UrlUtils::urlParametersToTuple2List)
                .getOrElse(Tuple2List.tuple2List());
    }

    public static StreamingUrlParams getCurrent() {
        return getCurrentO()
                .getOrElse(new StreamingUrlParams());
    }

    public static Option<StreamingUrlParams> getCurrentO() {
        return threadLocal.getO();
    }

    public static void runWith(StreamingUrlParams value, Runnable runnable) {
        threadLocal.runWith(value, runnable);
    }

    private static Tuple2List<String, String> getTuple2ListFromPairs(String... pairs) {
        return Tuple2List.fromPairs((Object[]) pairs);
    }
}
