package ru.yandex.chemodan.videostreaming.framework.hls.ffmpeg.receiver;

import java.io.InputStream;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Consumer;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.http.YandexCloudRequestIdHandler;
import ru.yandex.chemodan.videostreaming.framework.ffmpeg.FFmpegOutput;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsOutputProvider;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsSegmentMeta;
import ru.yandex.chemodan.videostreaming.framework.util.RequestIdHelper;
import ru.yandex.misc.io.InputStreamX;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class FFmpegReceiverManager implements HlsOutputProvider {
    @SuppressWarnings("unused")
    private static Logger logger = LoggerFactory.getLogger(FFmpegReceiverManager.class);

    private final int port;

    private final MapF<UUID, Session> sessions = Cf.x(new ConcurrentHashMap<>());

    public FFmpegReceiverManager(int port) {
        this.port = port;
    }

    @Override
    public FFmpegOutput getOutput(HlsSegmentMeta<?> segmentMeta, Consumer<InputStream> consumer) {
        Session session = startSession(segmentMeta.region.getIndex(), consumer);
        return FFmpegOutput.raw(
                // "fake=.ts" just to satisfy FFmpeg - it expects ".ts" at the end
                String.format("http://localhost:%d/ffmpeg/%s/s%%d.ts?%s=%s&fake=.ts",
                        port,
                        session.id.toString(),
                        YandexCloudRequestIdHandler.PARENT_RID_PARAM_NAME,
                        RequestIdHelper.getCurrentRequestId()
                ),
                session::close
        );
    }

    private Session startSession(int segmentNumber, Consumer<InputStream> consumer) {
        Session session = new Session(segmentNumber, consumer);
        sessions.put(session.id, session);
        return session;
    }

    public void handleFFmpegUpload(FFmpegReceiverUploadMeta uploadMeta, InputStream inputStream) {
        getConsumer(uploadMeta)
                .accept(inputStream);
    }

    private Consumer<InputStream> getConsumer(FFmpegReceiverUploadMeta uploadMeta) {
        Option<Session> sessionO = sessions.getO(uploadMeta.sessionId);
        if (sessionO.isMatch(s -> s.waitsFor(uploadMeta.filename))) {
            return sessionO.get()::consume;
        } else {
            return inputStream -> new InputStreamX(inputStream)
                    .readToDevNull();
        }
    }

    private class Session implements AutoCloseable {
        final UUID id = UUID.randomUUID();
        final int segmentNumber;
        final Consumer<InputStream> consumer;

        Session(int segmentNumber, Consumer<InputStream> consumer) {
            this.segmentNumber = segmentNumber;
            this.consumer = consumer;
        }

        boolean waitsFor(String filename) {
            return getExpectedSegmentFilename().equals(filename);
        }

        private String getExpectedSegmentFilename() {
            return String.format("s%d.ts", segmentNumber);
        }

        void consume(InputStream inputStream) {
            close();
            consumer.accept(inputStream);
        }

        @Override
        public void close() {
            sessions.removeTs(id);
        }
    }
}
