package ru.yandex.chemodan.videostreaming.framework.hls.framesegment;

import lombok.Value;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.videostreaming.framework.media.MediaInfo;
import ru.yandex.chemodan.videostreaming.framework.media.units.Fraction;
import ru.yandex.chemodan.videostreaming.framework.media.units.MediaTime;
import ru.yandex.chemodan.videostreaming.framework.media.units.SampleFrequency;

/**
 * @author Dmitriy Amelin (lemeh)
 */
@Value
public class FrameAwareStream {
    public static final SampleFrequency DEFAULT_SAMPLE_FREQ = new SampleFrequency(48000);

    Fraction frmDuration;

    Fraction segDuration;

    Fraction offset;

    public FrameAwareStream(Fraction frmDuration, Fraction segDuration) {
        this(frmDuration, segDuration, Fraction.ZERO);
    }

    public FrameAwareStream(Fraction frmDuration, Fraction segDuration, Fraction offset) {
        this.frmDuration = frmDuration;
        this.segDuration = segDuration;
        this.offset = offset;
    }

    public FrameAwareStream withOffsetO(Option<Fraction> offsetO) {
        return offsetO.map(offset -> new FrameAwareStream(frmDuration, segDuration, offset))
                .getOrElse(this);
    }

    public static FrameAwareStream fromStreamAndSegDuration(MediaInfo.VideoStream stream, MediaTime segDuration) {
        return fromFrmLenAndSegDuration(
                stream.getTranscodeFrameRate()
                        .toFraction()
                        .invert(),
                segDuration
        );
    }

    public static FrameAwareStream fromSampleFrequencyAndSegDuration(SampleFrequency sampleFreq, MediaTime segLen) {
        return fromFrmLenAndSegDuration(new Fraction(1024, sampleFreq.getValue()), segLen);
    }

    public static FrameAwareStream fromFrmLenAndSegDuration(Fraction frmLen, MediaTime segLen) {
        return new FrameAwareStream(
                frmLen.multiply(MediaTime.MICROS_PER_SECOND),
                new Fraction(segLen.getMicros(), 1)
        );
    }

    public Segment getSegment(int segNum) {
        Fraction start = calcFrameOffsetBeforeSegmentOffset(segNum);
        Fraction end = calcFrameOffsetBeforeSegmentOffset(segNum + 1);
        return new Segment(start, end).shift(offset);
    }

    public Segment getInnerSegmentExcludingRight(Segment outer) {
        Segment inner = getInnerSegment(outer);
        return !inner.getEndFrac().equals(outer.getEndFrac()) ? inner : inner.shrinkRight(frmDuration);
    }

    public Segment getInnerSegment(Segment segment) {
        return new Segment(
                calcFrameOffsetAfterTime(segment.getStartFrac()),
                calcFrameOffsetBeforeTime(segment.getEndFrac())
        );
    }

    private Fraction calcFrameOffsetBeforeSegmentOffset(int segNum) {
        Fraction segTime = segDuration.multiply(segNum);
        return calcFrameOffsetBeforeTime(segTime);
    }

    public Fraction calcFrameOffsetAfterTime(Fraction time) {
        return frmDuration.multiply(time.divide(frmDuration).ceil());
    }

    private Fraction calcFrameOffsetBeforeTime(Fraction time) {
        return frmDuration.multiply(time.divide(frmDuration).floor());
    }

    public MediaTime getFramesDuration(int frmCnt) {
        return new MediaTime(getFramesDurationFrac(frmCnt).floor());
    }

    public Fraction getFramesDurationFrac(int frmCnt) {
        return frmDuration.multiply(frmCnt);
    }

    @Value
    public static class Segment {
        final Fraction startFrac;

        final Fraction endFrac;

        static Segment fromStartAndDuration(Fraction start, Fraction duration) {
            return new Segment(start, start.plus(duration));
        }

        public Segment withStart(Fraction startFrac) {
            return new Segment(startFrac, endFrac);
        }

        public Segment withEnd(Fraction endFrac) {
            return new Segment(startFrac, endFrac);
        }

        public Segment shift(Fraction amount) {
            return new Segment(startFrac.plus(amount), endFrac.plus(amount));
        }

        public Segment expand(Fraction leftAmount, Fraction rightAmount) {
            return new Segment(
                    startFrac.minus(leftAmount.abs())
                            .zeroIfNegative(),
                    endFrac.plus(rightAmount.abs())
            );
        }

        public Segment shrinkRight(Fraction amount) {
            return withEnd(endFrac.minus(amount.abs()));
        }

        public MediaTime getStart() {
            return new MediaTime(startFrac.floor());
        }

        public MediaTime getEnd() {
            return new MediaTime(endFrac.floor());
        }

        public MediaTime getDuration() {
            return new MediaTime(getDurationFrac().floor());
        }

        private Fraction getDurationFrac() {
            return endFrac.minus(startFrac);
        }
    }
}
