package ru.yandex.chemodan.videostreaming.framework.hls.framesegment;

import lombok.Value;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.videostreaming.framework.media.MediaInfo;
import ru.yandex.chemodan.videostreaming.framework.media.units.Fraction;
import ru.yandex.chemodan.videostreaming.framework.media.units.MediaTime;

/**
 * @author Dmitriy Amelin (lemeh)
 */
@SuppressWarnings("unused")
public class FrameAwareStreamPair {
    final Option<FrameAwareStream> videoStreamO;

    final Option<FrameAwareStream> audioStreamO;

    public FrameAwareStreamPair(Option<FrameAwareStream> videoStreamO, Option<FrameAwareStream> audioStreamO) {
        this.videoStreamO = videoStreamO;
        this.audioStreamO = audioStreamO;
    }

    public static FrameAwareStreamPair fromMediaAndSegmentDuration(MediaInfo media, MediaTime segDuration) {
        Option<FrameAwareStream> audioStreamO =
                media.getAudioStreamO()
                        .map(mediaAudio ->
                                FrameAwareStream.fromSampleFrequencyAndSegDuration(
                                        FrameAwareStream.DEFAULT_SAMPLE_FREQ,
                                        segDuration
                                )
                        );
        Option<FrameAwareStream> videoStreamO =
                media.getVideoStreamO()
                        .map(mediaVideo -> FrameAwareStream.fromStreamAndSegDuration(mediaVideo, segDuration))
                        .map(videoStream -> videoStream.withOffsetO(audioStreamO.map(FrameAwareStream::getFrmDuration)));
        return new FrameAwareStreamPair(videoStreamO, audioStreamO);
    }

    public Fraction getAudioFrmDuration() {
        return audioStreamO.map(FrameAwareStream::getFrmDuration)
                .getOrElse(Fraction.ZERO);
    }

    public FrameAwareStream getMain() {
        return getMainO().get();
    }

    public Option<FrameAwareStream> getMainO() {
        return videoStreamO.orElse(audioStreamO);
    }

    public FrameAwareStream getVideo() {
        return getVideoO().get();
    }

    public Option<FrameAwareStream> getVideoO() {
        return videoStreamO;
    }

    public FrameAwareStream getAudio() {
        return getAudioO().get();
    }

    public Option<FrameAwareStream> getAudioO() {
        return audioStreamO;
    }

    public Segment getSegment(int segNum) {
        Option<FrameAwareStream.Segment> videoSegO = videoStreamO.map(videoStream -> videoStream.getSegment(segNum));
        Option<FrameAwareStream.Segment> audioSegO =
                audioStreamO.map(audioStream -> getAudioSegment(segNum, audioStream, videoSegO));
        return new Segment(videoSegO, audioSegO, segNum);
    }

    private FrameAwareStream.Segment getAudioSegment(int segNum, FrameAwareStream audioStream,
            Option<FrameAwareStream.Segment> videoSegO)
    {
        return videoSegO.map(videoSeg -> getAudioSegment(segNum, audioStream, videoSeg))
                .getOrElse(() -> audioStream.getSegment(segNum));
    }

    private FrameAwareStream.Segment getAudioSegment(int segNum, FrameAwareStream audioStream,
            FrameAwareStream.Segment videoSeg)
    {
        FrameAwareStream.Segment audioSeg = audioStream.getInnerSegmentExcludingRight(videoSeg);
        return segNum != 0 ? audioSeg : audioSeg.withStart(Fraction.ZERO);
    }

    @Value
    public static class Segment {
        Option<FrameAwareStream.Segment> videoO;

        Option<FrameAwareStream.Segment> audioO;

        int segNum;

        public FrameAwareStream.Segment getMain() {
            return getMainO().get();
        }

        public Option<FrameAwareStream.Segment> getMainO() {
            return videoO.orElse(audioO);
        }

        public FrameAwareStream.Segment getVideo() {
            return videoO.get();
        }

        public FrameAwareStream.Segment getAudio() {
            return audioO.get();
        }

        public MediaTime getStart() {
            return Option.when(segNum == 0, MediaTime.ZERO)
                    .orElse(getMainO().map(FrameAwareStream.Segment::getStart))
                    .get();
        }

        public MediaTime getDuration() {
            return getMainO()
                    .map(FrameAwareStream.Segment::getDuration)
                    .get();
        }

        public boolean hasAudioOffset() {
            return segNum != 0 && videoO.isPresent() && audioO.isPresent();
        }
    }

    private static MediaTime calcStartDifference(FrameAwareStream.Segment videoSeg, FrameAwareStream.Segment audioSeg) {
        return new MediaTime(
                audioSeg.getStartFrac()
                        .minus(videoSeg.getStartFrac())
                        .floor()
        );
    }
}
