package ru.yandex.chemodan.videostreaming.framework.hls.segmentprovider;

import java.io.IOException;
import java.io.InputStream;
import java.util.Optional;
import java.util.function.Consumer;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.util.ComparableUtils;
import ru.yandex.chemodan.videostreaming.framework.ffmpeg.FFmpeg;
import ru.yandex.chemodan.videostreaming.framework.ffmpeg.FFmpegStatsHandler;
import ru.yandex.chemodan.videostreaming.framework.hls.AudioSegmentId3HeaderUtil;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsSegmentMeta;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsStreamQuality;
import ru.yandex.chemodan.videostreaming.framework.hls.StreamingParams;
import ru.yandex.chemodan.videostreaming.framework.hls.segmentprovider.argfiller.FFmpegFillParams;
import ru.yandex.chemodan.videostreaming.framework.hls.segmentprovider.caching.CacheData;
import ru.yandex.chemodan.videostreaming.framework.hls.segmentprovider.caching.HlsSegmentCache;
import ru.yandex.chemodan.videostreaming.framework.hls.segmentprovider.prefetch.HlsSegmentPrefetch;
import ru.yandex.chemodan.videostreaming.framework.hls.segmentprovider.prefetch.HlsSegmentPrefetchTrigger;
import ru.yandex.chemodan.videostreaming.framework.hls.segmentprovider.prefetch.SegmentPrefetchFFmpegStatsHandler;
import ru.yandex.chemodan.videostreaming.framework.hls.stats.FFUtilStats;
import ru.yandex.chemodan.videostreaming.framework.util.AsyncCacheableInputStreamSource;
import ru.yandex.chemodan.videostreaming.framework.util.CommonThreadPoolHolder;
import ru.yandex.chemodan.videostreaming.framework.web.HlsErrorSource;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.cache.Cache;
import ru.yandex.misc.cache.CacheUtils;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.InputStreamX;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class HlsStreamingManager<T> {
    private static final Logger logger = LoggerFactory.getLogger(HlsStreamingManager.class);

    private static final double WAIT_COEFF = 2.2;

    private static final Duration WAIT_SLEEP_DURATION = new Duration(200);

    private static final int INITIAL_SEGMENT_BUFFER_SIZE = (int) DataSize.fromMegaBytes(1).toBytes();

    public static final Consumer<InputStream> devNullInputStreamConsumer = is -> new InputStreamX(is).readToDevNull();

    private final FFmpeg ffmpeg;

    private final HlsFFmpegCommandProvider<T> commandProvider;

    private final HlsSegmentCache<T> cache;

    private final HlsSegmentPrefetch<T> segmentPrefetch;

    private final HlsSegmentPrefetchTrigger segmentPrefetchTrigger;

    private final StreamingParams streamingParams;

    private final Cache<HlsSegmentMeta<T>, AsyncCacheableInputStreamSource> prefetchCache;

    public HlsStreamingManager(FFmpeg ffmpeg, HlsFFmpegCommandProvider<T> commandProvider, HlsSegmentCache<T> cache,
            HlsSegmentPrefetch<T> segmentPrefetch,
            HlsSegmentPrefetchTrigger segmentPrefetchTrigger,
            StreamingParams streamingParams,
            int prefetchCacheSize)
    {
        this.ffmpeg = ffmpeg;
        this.commandProvider = commandProvider;
        this.cache = cache;
        this.segmentPrefetch = segmentPrefetch;
        this.segmentPrefetchTrigger = segmentPrefetchTrigger;
        this.streamingParams = streamingParams;
        this.prefetchCache = CacheUtils.newLru(prefetchCacheSize);
    }

    public void prefetch(HlsSegmentMeta<T> segmentMeta) {
        Streamer streamer = new Streamer(segmentMeta);
        prefetchCache.putInCache(segmentMeta, streamer.iss);

        boolean prepared = cache.prepare(segmentMeta);
        if (!prepared) {
            return;
        }

        streamer.startTranscoding()
                .streamToCache(true);
    }

    public void streamLocalPrefetch(HlsSegmentMeta<T> segmentMeta, Consumer<InputStream> consumer) {
        Optional<AsyncCacheableInputStreamSource> issO = prefetchCache.getFromCache(segmentMeta);
        if (!issO.isPresent()) {
            throw new LocalPrefetchNotFoundException();
        }

        try(InputStream input = issO.get().getInput()) {
            consumer.accept(input);
        } catch (IOException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    public HlsSegmentSource stream(HlsSegmentMeta<T> segmentMeta, Consumer<InputStream> consumer) {
        if (streamingParams.isCacheDisabled()) {
            return new Streamer(segmentMeta, consumer)
                    .streamFromFFmpeg();
        }

        logger.info("Prefetching segments using prefetch size = {}", streamingParams.getInitPrefetchSize());
        for(int n = 1; n <= streamingParams.getInitPrefetchSize(); n++) {
            Option<HlsSegmentMeta<T>> prefetchSegmentMetaO = segmentMeta.getSiblingIfExistsO(n);
            if (prefetchSegmentMetaO.isPresent()) {
                HlsSegmentMeta<T> prefetchSegmentMeta = prefetchSegmentMetaO.get();
                CacheData cacheData = cache.getCacheData(prefetchSegmentMeta);
                if (cacheData.isNotEmpty()) {
                    continue;
                }

                CommonThreadPoolHolder.runAsync(() -> segmentPrefetch.prefetch(prefetchSegmentMeta));
            }
        }

        FFmpegStatsHandler statsHandler;
        if (!streamingParams.isDisableDynamicPrefetch()) {
            statsHandler = new SegmentPrefetchFFmpegStatsHandler<>(
                    segmentPrefetch,
                    segmentMeta,
                    segmentPrefetchTrigger,
                    streamingParams.getInitPrefetchSize(),
                    streamingParams.getMaxPrefetchSize()
            );
        } else {
            statsHandler = FFmpegStatsHandler.noop;
        }
        return new Streamer(segmentMeta, consumer, statsHandler)
                .streamFromCacheOrFFmpeg();
    }

    private class Streamer {
        final HlsSegmentMeta<T> segmentMeta;

        final FFmpegStatsHandler statsHandler;

        final AsyncCacheableInputStreamSource iss = new AsyncCacheableInputStreamSource(INITIAL_SEGMENT_BUFFER_SIZE);

        final Consumer<InputStream> consumer;

        final CacheHelper cacheHelper = new CacheHelper();

        Option<FFmpeg.Session> ffmpegSession = Option.empty();

        Option<FFmpegFillParams> ffmpegFillParams = Option.empty();

        Streamer(HlsSegmentMeta<T> segmentMeta) {
            this(segmentMeta, FFmpegStatsHandler.noop);
        }

        Streamer(HlsSegmentMeta<T> segmentMeta, FFmpegStatsHandler statsHandler) {
            this(segmentMeta, devNullInputStreamConsumer, statsHandler);
        }

        Streamer(HlsSegmentMeta<T> segmentMeta, Consumer<InputStream> consumer) {
            this(segmentMeta, consumer, FFmpegStatsHandler.noop);
        }

        Streamer(HlsSegmentMeta<T> segmentMeta, Consumer<InputStream> consumer, FFmpegStatsHandler statsHandler) {
            this.segmentMeta = segmentMeta;
            this.consumer = consumer;
            this.statsHandler = statsHandler;
        }

        Streamer startTranscoding() {
            FFmpegCommandAndFillParams commandAndFillParams = commandProvider.get(segmentMeta, iss::receiveFrom);
            ffmpegFillParams = commandAndFillParams.getFillParams();
            ffmpegSession = Option.of(
                    ffmpeg.execute(commandAndFillParams.getCommand(), statsHandler, iss::setExternalException)
            );
            return this;
        }

        HlsSegmentSource streamFromFFmpeg() {
            startTranscoding();
            consumer.accept(getInputStream());
            FFUtilStats.getCurrent().setFFmpegAppCpuTimeO(ffmpegSession.get().getAppCpuTimeO());
            return HlsSegmentSource.FFMPEG;
        }

        InputStream getInputStream() {
            if (id3HeaderPrependNeeded()) {
                return AudioSegmentId3HeaderUtil.prependTo(iss.getInputStreamX(), ffmpegFillParams.get().getOffset());
            } else {
                return iss.getInputStreamX();
            }
        }

        void streamToCache(boolean prefetch) {
            try {
                cache.put(segmentMeta, getFinalIss());
            } catch (RuntimeException e) {
                logger.error("Error while {} segment#{}",
                        prefetch ? "saving prefetch to cache" : "caching",
                        segmentMeta.getIndex(),
                        e
                );
            }
        }

        private boolean id3HeaderPrependNeeded() {
            return segmentMeta.getQuality().equals(HlsStreamQuality._AAC) && ffmpegFillParams.isPresent();
        }

        private InputStreamSource getFinalIss() {
            return new InputStreamSource() {
                @Override
                public InputStream getInput() {
                    return getInputStream();
                }
            };
        }

        HlsSegmentSource streamFromCache() {
            CacheData cacheData = cacheHelper.memoized();
            try(InputStream in = cacheData.getInput()) {
                consumer.accept(in);
            } catch (IOException e) {
                throw ExceptionUtils.translate(e);
            }
            return cacheData.getSource().toSegmentSource();
        }

        HlsSegmentSource streamFromFFmpegAndCache() {
            streamFromFFmpeg();
            CommonThreadPoolHolder.runAsync(this::putToCacheIfPossible);
            return HlsSegmentSource.FFMPEG;
        }

        HlsSegmentSource streamFromCacheOrFFmpeg() {
            if (cacheHelper.mostRecent(getFetchSegmentPolicy()).isPresent()) {
                return streamFromCache();
            }

            if (cacheHelper.memoized().isInProgress()) {
                Instant endOfWait = Instant.now().plus(getMaxCacheWait());
                while(Instant.now().isBefore(endOfWait)) {
                    if (cacheHelper.mostRecent().isPresent()) {
                        return streamFromCache();
                    }

                    try {
                        Thread.sleep(WAIT_SLEEP_DURATION.getMillis());
                    } catch (InterruptedException e) {
                        break;
                    }
                }

                logger.warn("Unable to receive segment from cache - receiving from FFmpeg");
            }

            return streamFromFFmpegAndCache();
        }

        private MasterSlavePolicy getFetchSegmentPolicy() {
            return streamingParams.isFetchSegmentFromMaster() ? MasterSlavePolicy.R_MS : MasterSlavePolicy.R_SM;
        }

        private Duration getMaxCacheWait() {
            return ComparableUtils.min(
                    Duration.millis((long) WAIT_COEFF * segmentMeta.getDuration().getMillis()),
                    streamingParams.getMaxCacheWait()
            );
        }

        void putToCacheIfPossible() {
            if (!cacheHelper.tryPrepare()) {
                return;
            }

            streamToCache(false);
        }

        class CacheHelper {
            Option<CacheData> cacheDataO = Option.empty();

            boolean tryPrepare() {
                return memoized().isInProgress() || cache.prepare(segmentMeta);
            }

            CacheData memoized() {
                return cacheDataO.getOrElse(this::mostRecent);
            }

            CacheData mostRecent() {
                cacheDataO = Option.of(cache.getCacheData(segmentMeta));
                return cacheDataO.get();
            }

            CacheData mostRecent(MasterSlavePolicy policy) {
                return MasterSlaveContextHolder.withPolicy(policy, (Function0<CacheData>) this::mostRecent);
            }
        }
    }

    private static final class LocalPrefetchNotFoundException extends RuntimeException implements HlsErrorSource {
        private static final HlsError ERROR =
                new HlsError(HttpStatus.SC_404_NOT_FOUND, "Requested prefetched segment not found");

        @Override
        public HlsError getHlsError() {
            return ERROR;
        }
    }
}
