package ru.yandex.chemodan.videostreaming.framework.hls.segmentprovider;

import java.io.ByteArrayOutputStream;

import org.joda.time.Duration;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.junit.MockitoJUnit;
import org.mockito.junit.MockitoRule;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.videostreaming.framework.ffmpeg.FFmpeg;
import ru.yandex.chemodan.videostreaming.framework.ffmpeg.FFmpegCommand;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsSegmentMeta;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsStreamQuality;
import ru.yandex.chemodan.videostreaming.framework.hls.StreamingParams;
import ru.yandex.chemodan.videostreaming.framework.hls.segmentprovider.caching.CacheData;
import ru.yandex.chemodan.videostreaming.framework.hls.segmentprovider.caching.HlsSegmentCache;
import ru.yandex.chemodan.videostreaming.framework.media.units.MediaTime;
import ru.yandex.misc.io.ByteArrayInputStreamSource;
import ru.yandex.misc.io.IoUtils;
import ru.yandex.misc.test.Assert;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class HlsStreamingManagerTest {
    private static final String CACHE_DATA_PAYLOAD = "cached data";

    @Rule
    public MockitoRule rule = MockitoJUnit.rule();

    @Mock
    private HlsSegmentCache<Object> segmentCacheMock;

    @Mock
    private HlsFFmpegCommandProvider<Object> commandProvider;

    @Mock
    private HlsSegmentMeta<Object> cacheSegmentMeta;

    @Mock
    private HlsSegmentMeta<Object> transcodingSegmentMeta;

    @Mock
    private HlsSegmentMeta<Object> inProgressSegmentMeta;

    @Mock
    private FFmpeg ffmpeg;

    @Before
    public void initMocks() {
        Mockito.when(segmentCacheMock.getCacheData(transcodingSegmentMeta))
                .thenReturn(CacheData.ABSENT);

        Mockito.when(segmentCacheMock.getCacheData(cacheSegmentMeta))
                .thenReturn(consCacheData());

        Mockito.when(segmentCacheMock.getCacheData(inProgressSegmentMeta))
                .thenReturn(CacheData.IN_PROGRESS, CacheData.IN_PROGRESS, consCacheData());

        Mockito.when(callTranscoder(ffmpeg))
                .thenReturn(Mockito.mock(FFmpeg.Session.class));

        Mockito.when(cacheSegmentMeta.getQuality())
                .thenReturn(HlsStreamQuality._1080P);

        Mockito.when(transcodingSegmentMeta.getQuality())
                .thenReturn(HlsStreamQuality._1080P);

        Mockito.when(inProgressSegmentMeta.getQuality())
                .thenReturn(HlsStreamQuality._1080P);

        Mockito.when(commandProvider.get(Mockito.any(), Mockito.any()))
                .thenReturn(new FFmpegCommandAndFillParams(Mockito.mock(FFmpegCommand.class), Option.empty()));
    }

    @Test
    public void verifyThatTranscodingWasStartedWhenCacheIsMissing() {
        consSut().stream(transcodingSegmentMeta, inputStream -> { /* do nothing */ });
        verifyTranscoderCalled();
    }

    @Test
    public void verifyThatDataWasStreamedFromCacheWhenCacheIsPresent() {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        consSut().stream(cacheSegmentMeta, inputStream -> IoUtils.copy(inputStream, out));
        Assert.equals(CACHE_DATA_PAYLOAD, new String(out.toByteArray()));
        verifyTranscoderWasNotCalled();
    }

    @Test
    public void verifyThatDataWasStreamedFromCacheWhenCacheFillIsFast() {
        Mockito.when(inProgressSegmentMeta.getDuration())
                .thenReturn(MediaTime.seconds(5));

        ByteArrayOutputStream out = new ByteArrayOutputStream();
        consSut().stream(inProgressSegmentMeta, inputStream -> IoUtils.copy(inputStream, out));
        Assert.equals(CACHE_DATA_PAYLOAD, new String(out.toByteArray()));
        verifyTranscoderWasNotCalled();
    }

    @Test
    public void verifyThatTranscodingWasStartedWhenCacheFillIsSlow() {
        Mockito.when(inProgressSegmentMeta.getDuration())
                .thenReturn(MediaTime.seconds(5));

        consSut(new Duration(100)).stream(inProgressSegmentMeta, inputStream -> { /* do nothing */ });
        verifyTranscoderCalled();
    }

    private HlsStreamingManager<Object> consSut() {
        return consSut(Duration.standardSeconds(1));
    }

    private HlsStreamingManager<Object> consSut(Duration maxCacheWait) {
        return new HlsStreamingManager<>(
                ffmpeg, commandProvider, segmentCacheMock, null, null,
                new StreamingParams(MediaTime.seconds(5), 0, 0, maxCacheWait),
                1);
    }

    private static CacheData consCacheData() {
        return CacheData.fromCache(new ByteArrayInputStreamSource(CACHE_DATA_PAYLOAD.getBytes()));
    }

    private void verifyTranscoderCalled() {
        callTranscoder(Mockito.verify(ffmpeg));
    }

    private void verifyTranscoderWasNotCalled() {
        callTranscoder(Mockito.verify(ffmpeg, Mockito.never()));
    }

    private static FFmpeg.Session callTranscoder(FFmpeg ffmpeg) {
        return ffmpeg.execute(Mockito.any(), Mockito.any(), Mockito.any());
    }
}
