package ru.yandex.chemodan.videostreaming.framework.hls.segmentprovider.argfiller;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.videostreaming.framework.ffmpeg.FFmpegCommandBuilder;
import ru.yandex.chemodan.videostreaming.framework.ffmpeg.FFmpegTimeDurationUtil;
import ru.yandex.chemodan.videostreaming.framework.hls.ffmpeg.transcoding.FFmpegMapping;
import ru.yandex.chemodan.videostreaming.framework.hls.ffmpeg.transcoding.HlsFFmpegCommandContext;
import ru.yandex.chemodan.videostreaming.framework.media.units.FrameRate;
import ru.yandex.chemodan.videostreaming.framework.media.units.MediaTime;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class DefaultFFmpegArgFiller {
    public static final FFmpegArgFiller DEFAULT =
            (cmd, ctx) -> new DefaultFFmpegArgFiller(cmd, ctx, false).fill();

    public static final FFmpegArgFiller EXPERIMENT =
            (cmd, ctx) -> new DefaultFFmpegArgFiller(cmd, ctx, true).fill();

    public static FFmpegArgFiller get(boolean experimentEnabled) {
        return experimentEnabled ? EXPERIMENT : DEFAULT;
    }

    private static final FrameRate DEFAULT_FRAME_RATE = new FrameRate(25);

    private static final FrameRate MAX_FRAME_RATE = new FrameRate(60);

    private final FFmpegCommandBuilder builder;

    private final HlsFFmpegCommandContext ctx;

    private final boolean experimentEnabled;

    private DefaultFFmpegArgFiller(FFmpegCommandBuilder builder, HlsFFmpegCommandContext ctx,
            boolean experimentEnabled)
    {
        this.builder = builder;
        this.ctx = ctx;
        this.experimentEnabled = experimentEnabled;
    }

    public Option<FFmpegFillParams> fill() {
        fillCommon();
        if (experimentEnabled) {
            Option<FrameRate> frameRateO = getForcedFrameRateO();
            builder
                    .inIf("-vsync", "passthrough", !frameRateO.isPresent())
                    .outO("-r", frameRateO.map(FrameRate::getValue))
//                        .out("-vf",
//                                "drawtext=fontfile=/usr/share/fonts/truetype/dejavu/DejaVuSans.ttf"
//                                        + ":text=Experiment:fontcolor=white:fontsize=100"
//                                        + ":box=1:boxcolor=black@0.5:boxborderw=5"
//                                        + ":x=(w-text_w)/2:y=(h-text_h)/2"
//                        )
            ;
        } else {
            builder.out("-r", 25.0);
        }
        return Option.empty();
    }

    private void fillCommon() {
        builder
                // simplified 1 parameter audio timestamp matching
                // 0(disabled), 1(filling and trimming), >1(maximum stretch/squeeze in samples per second)
                .inIf("-async", "1", ctx.isNotFirstSegment())

                .inIf("-gapless", ctx.isUseGapless())

                // MUST NOT be set for first segment!
                // see https://wiki.yandex-team.ru/disk/videostreaming/ffmpeg-params/#param-ss
                .inIf("-ss", formatTimeDuration(ctx.getSegmentStart()), ctx.isNotFirstSegment())

                .in("-t", formatTimeDuration(ctx.getOutDuration()))

                .in("-threads", ctx.getInThreads())

//                .add("-seekable", 1) // Control seekability of connection.
//                .add("-multiple_requests", 1) // Use persistent connections if set to 1, default is 0

                // https://wiki.yandex-team.ru/disk/videostreaming/ffmpeg-params/#param-recv_buffer_size
                .in("-recv_buffer_size", ctx.getRecvBufferSize().toBytes())

                .out("-threads", ctx.getOutThreads())

                .applyIf(this::addVideoParams, ctx.hasVideo())

                .outO(ctx.getMappingO().map(FFmpegMapping::getArgs))

                .applyIf(this::addVideoCodecParams, ctx.hasVideo())
                .elseOut("-vn")

                .applyIf(this::addAudioCodecParams, ctx.hasAudio())
                .elseOut("-an")

                .out("-sn")

                .out("-f", "ssegment")
                .out("-segment_format", "mpegts")
                .out("-segment_start_number", ctx.getSegmentStartNumber().toString())
                .out("-initial_offset", FFmpegTimeDurationUtil.formatInSeconds(ctx.getSegmentStart()))
                .out("-segment_time", FFmpegTimeDurationUtil.formatInSeconds(ctx.getSegmentDuration()));
    }

    private Option<FrameRate> getForcedFrameRateO() {
        if (!ctx.getFrameRateO().isPresent()) {
            return Option.of(DEFAULT_FRAME_RATE);
        }

        FrameRate frameRate = ctx.getFrameRateO().get();
        return Option.when(frameRate.gt(MAX_FRAME_RATE), MAX_FRAME_RATE);
    }

    private void addVideoParams(FFmpegCommandBuilder builder) {
        builder.out("-s", ctx.getDimension()) // set frame size (WxH or abbreviation)
                .out("-force_key_frames",
                        String.format("expr:if(isnan(prev_forced_t),gte(t,%s),gte(t,prev_forced_t+%s))",
                                FFmpegTimeDurationUtil.formatInSeconds(MediaTime.ZERO),
                                FFmpegTimeDurationUtil.formatInSeconds(ctx.getSegmentDuration())
                        )
                );
    }

    private void addVideoCodecParams(FFmpegCommandBuilder builder) {
        builder.out("-b:v", "" + ctx.getVideoBitRate().getKbps() + "k")
                .out("-bufsize", "" + Math.max(1835, ctx.getVideoBitRate().getKbps() * 2) + "k")
                .out("-maxrate", "" + ctx.getVideoBitRate().getKbps() * 6 / 5 + "k")

                .out("-vcodec", "h264")
                .out("-profile:v", "baseline")
                .out("-level", "3.0")
                .out("-pix_fmt", "yuv420p")

                .out("-preset", "veryfast")

//                .out("-sc_threshold", 0)
        ;
    }

    private void addAudioCodecParams(FFmpegCommandBuilder builder) {
        builder.out("-b:a", ctx.getAudioBitRateO().get().getKbps() + "k")
                .out("-acodec", "aac")
                .out("-strict", "-2");
    }

    private static String formatTimeDuration(MediaTime seekPosition) {
        return FFmpegTimeDurationUtil.format(seekPosition);
    }
}
