package ru.yandex.chemodan.videostreaming.framework.hls.videoinfo;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsResource;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsStreamQuality;
import ru.yandex.chemodan.videostreaming.framework.media.units.MediaTime;
import ru.yandex.chemodan.videostreaming.framework.util.MediaTimeAsMillisMarshaller;
import ru.yandex.chemodan.videostreaming.framework.util.MediaTimeAsMillisUnmarshaller;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.MembersToBind;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.bender.config.BenderConfiguration;
import ru.yandex.misc.bender.config.CustomMarshallerUnmarshallerFactoryBuilder;
import ru.yandex.misc.bender.serialize.BenderSerializer;
import ru.yandex.misc.digest.Sha256;
import ru.yandex.misc.image.Dimension;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.lang.DefaultObject;

/**
 * @author vavinov
 */
@Bendable
public class VideoInfo {
    private static final BenderSerializer<VideoInfo> SERIALIZER = Bender.serializer(VideoInfo.class,
            BenderConfiguration.cons(
                    MembersToBind.WITH_ANNOTATIONS, false,
                    CustomMarshallerUnmarshallerFactoryBuilder.cons()
                            .add(MediaTime.class,
                                    new MediaTimeAsMillisMarshaller(),
                                    new MediaTimeAsMillisUnmarshaller())
                            .build())
    );

    @BenderPart(name = "stream_id", strictName = true)
    public final String streamId;

    @BenderPart(name = "duration", strictName = true)
    public final MediaTime duration;

    @BenderPart(name = "total", strictName = true)
    public final int total;

    @BenderPart(name = "items", strictName = true)
    public final ListF<Playlist> playlists;

    @BenderPart(name = "orig_dimension", strictName = true)
    public final Option<Dimension> origDimension;

    public VideoInfo(
            String streamId,
            MediaTime duration,
            ListF<Playlist> playlists,
            Option<Dimension> origDimension)
    {
        this.streamId = streamId;
        this.duration = duration;
        this.total = playlists.size();
        this.playlists = playlists;
        this.origDimension = origDimension;
    }

    public static VideoInfo cons(
            HlsResource resource,
            String masterPlaylistUrl,
            Function<HlsStreamQuality, String> playlistUrlF)
    {
        MediaTime duration = resource.getTotalDurationO().getOrElse(MediaTime.ZERO);
        return new VideoInfo(
                Sha256.A.digest(masterPlaylistUrl).base64(),
                duration,
                resource.getStreams()
                        .map(stream -> Playlist.consPlaylist(stream, playlistUrlF))
                        .plus1(Playlist.consMasterPlaylist(masterPlaylistUrl)),
                resource.getRotatedDimensionO()
        );
    }

    public byte[] toJson() {
        return SERIALIZER.serializeJson(this);
    }

    @Bendable
    @SuppressWarnings("unused")
    private static class Playlist extends DefaultObject {
        @BenderPart(name = "resolution", strictName = true)
        public final String quality;

        @BenderPart(name = "links", strictName = true)
        public final MapF<String, String> playlistUrls;

        @BenderPart(name = "video_codec", strictName = true)
        public final String videoCodec = "H.264";

        @BenderPart(name = "audio_codec", strictName = true)
        public final String audioCodec = "AAC";

        @BenderPart(name = "container", strictName = true)
        public final String container = "hls";

        @BenderPart(name = "width", strictName = true)
        public final Option<Integer> width;

        @BenderPart(name = "height", strictName = true)
        public final Option<Integer> height;

        Playlist(String quality, String playlistUrl, Option<Dimension> dimension) {
            this.quality = quality;
            this.playlistUrls = Cf.map(extractScheme(playlistUrl), playlistUrl);
            this.width = dimension.map(Dimension::getWidth);
            this.height = dimension.map(Dimension::getHeight);
        }

        static Playlist consMasterPlaylist(String playlistUrl) {
            return new Playlist("adaptive", playlistUrl, Option.empty());
        }

        static Playlist consPlaylist(HlsResource.Stream stream,
                Function<HlsStreamQuality, String> playlistUrlF)
        {
            return new Playlist(
                    stream.getQuality().toRequestParamValue(),
                    playlistUrlF.apply(stream.getQuality()),
                    Option.of(stream.getDimension())
            );
        }

        static String extractScheme(String playlistUrl) {
            return new UriBuilder(playlistUrl)
                    .build()
                    .getScheme()
                    .toLowerCase();
        }
    }
}
