package ru.yandex.chemodan.videostreaming.framework.ignite;

import java.util.concurrent.atomic.AtomicInteger;

import org.joda.time.Duration;
import org.joda.time.Instant;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class ExponentialBackoff {
    private final Duration initialDelay;

    private final double multiplier;

    private final boolean resetOnSuccess;

    private final AtomicInteger attemptCount = new AtomicInteger();

    private volatile Instant lastAttemptAt = Instant.now();

    private volatile boolean skippedOrErrored = false;

    public ExponentialBackoff(Duration initialDelay, double multiplier) {
        this(initialDelay, multiplier, true);
    }

    private ExponentialBackoff(Duration initialDelay, double multiplier, boolean resetOnSuccess) {
        this.initialDelay = initialDelay;
        this.multiplier = multiplier;
        this.resetOnSuccess = resetOnSuccess;
    }

    public ExponentialBackoff doNotResetOnSuccess() {
        return new ExponentialBackoff(initialDelay, multiplier, false);
    }

    public boolean executeIfDelayIsOver(Runnable runnable) {
        if (delayIsNotOver()) {
            skippedOrErrored = true;
            return false;
        }

        if (!skippedOrErrored) {
            attemptCount.set(0);
        }

        try {
            runnable.run();
        } catch (RuntimeException ex) {
            skippedOrErrored = true;
            throw ex;
        } finally {
            attemptCount.incrementAndGet();
            lastAttemptAt = Instant.now();
        }

        if (isResetOnSuccess()) {
            attemptCount.set(0);
        }

        return true;
    }

    private boolean delayIsNotOver() {
        return attemptCount.get() != 0 && Instant.now().isBefore(delayUntil());
    }

    private Instant delayUntil() {
        return lastAttemptAt.plus(getDelay());
    }

    Duration getDelay() {
        if (attemptCount.get() == 0) {
            return Duration.ZERO;
        }

        return new Duration((int) (initialDelay.getMillis() * Math.pow(multiplier, attemptCount.get() - 1)));
    }

    public boolean isResetOnSuccess() {
        return resetOnSuccess;
    }
}
