package ru.yandex.chemodan.videostreaming.framework.ignite;

import java.util.function.Consumer;

import lombok.Builder;
import lombok.Value;
import org.joda.time.Duration;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.test.Assert;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class ExponentialBackoffTest {
    private static final Consumer<Integer> THROW_AT_FIRST_ATTEMPT_EXEC = n -> {
        if (n == 0) {
            throw new TestExecutionException();
        }
    };

    private static final Runnable THROW_EXEC = () -> {
        throw new TestExecutionException();
    };

    private static final Runnable OK_EXEC = () -> {
        /* do nothing - imitate successful execution */
    };

    @Test
    public void neverSkipExecutionForFirstAttempt() {
        Tester.builder()
                .execCount(3)
                .build()
                .exerciseSutAndAssert(Result.SUCCESS, Result.SUCCESS, Result.SUCCESS);
    }

    @Test
    public void skipExecutionIfDelayIsNotOver() {
        Tester.builder()
                .initialDelay(Duration.standardHours(1))
                .execCount(2)
                .resetOnSuccess(false)
                .execF(THROW_AT_FIRST_ATTEMPT_EXEC)
                .build()
                .exerciseSutAndAssert(Result.ERROR, Result.SKIPPED);
    }

    @Test
    public void executeIfDelayIsOver() {
        Tester.builder()
                .initialDelay(new Duration(1))
                .execCount(2)
                .resetOnSuccess(false)
                .sleepBetweenAttempts(10)
                .execF(THROW_AT_FIRST_ATTEMPT_EXEC)
                .build()
                .exerciseSutAndAssert(Result.ERROR, Result.SUCCESS);
    }

    @Test
    public void delayMustBeZeroBeforeAnyAttempt() {
        Assert.equals(Duration.ZERO, new ExponentialBackoff(Duration.millis(1), 2).getDelay());
    }

    @Test
    public void delayMustBeZeroAfterSuccessfulAttempt() {
        ExponentialBackoff sut = new ExponentialBackoff(Duration.millis(1), 2);
        sut.executeIfDelayIsOver(OK_EXEC);
        Assert.equals(Duration.ZERO, sut.getDelay());
    }

    @Test
    public void delayGrowsExponentially() {
        ExponentialBackoff sut = new ExponentialBackoff(Duration.millis(1), 2);

        sleepExerciseThrowAndAssertDelayEquals(sut, Duration.millis(1));
        sleepExerciseThrowAndAssertDelayEquals(sut, Duration.millis(2));
        sleepExerciseThrowAndAssertDelayEquals(sut, Duration.millis(4));
        sleepExerciseThrowAndAssertDelayEquals(sut, Duration.millis(8));
    }

    private void sleepExerciseThrowAndAssertDelayEquals(ExponentialBackoff sut, Duration expectedDelay) {
        sleep(sut.getDelay());
        try {
            sut.executeIfDelayIsOver(THROW_EXEC);
        } catch (TestExecutionException ex) {
            // ignore
        }
        Assert.equals(expectedDelay, sut.getDelay());
    }

    private static void sleep(Duration delay) {
        try {
            Thread.sleep(delay.getMillis());
        } catch (InterruptedException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    @Value
    @Builder(builderClassName = "Builder")
    private static class Tester {
        @lombok.Builder.Default
        Duration initialDelay = Duration.standardSeconds(1);

        @lombok.Builder.Default
        int execCount = 1;

        @lombok.Builder.Default
        long sleepBetweenAttempts = 0;

        @lombok.Builder.Default
        Consumer<Integer> execF = n -> { /* do nothing */ };

        @lombok.Builder.Default
        boolean resetOnSuccess = true;

        void exerciseSutAndAssert(Result... expected) {
            ExponentialBackoff sut = new ExponentialBackoff(initialDelay, 1.1);
            if (!resetOnSuccess) {
                sut = sut.doNotResetOnSuccess();
            }
            ListF<Result> results = Cf.arrayList();
            for (int i = 0; i < execCount; i++) {
                //noinspection ConstantConditions
                if (i != 0 && sleepBetweenAttempts != 0) {
                    try {
                        Thread.sleep(sleepBetweenAttempts);
                    } catch (InterruptedException e) {
                        throw ExceptionUtils.translate(e);
                    }
                }

                int attemptNumber = i;
                try {
                    boolean executed = sut.executeIfDelayIsOver(() -> execF.accept(attemptNumber));
                    results.add(executed ? Result.SUCCESS : Result.SKIPPED);
                } catch (TestExecutionException ex) {
                    results.add(Result.ERROR);
                }
            }
            Assert.equals(Cf.list(expected), results);
        }
    }

    enum Result {
        SKIPPED, SUCCESS, ERROR
    }

    private static class TestExecutionException extends RuntimeException {

    }
}
