package ru.yandex.chemodan.videostreaming.framework.ignite;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.Collection;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Supplier;

import org.apache.ignite.DataRegionMetrics;
import org.apache.ignite.DataRegionMetricsAdapter;
import org.apache.ignite.DataStorageMetrics;
import org.apache.ignite.DataStorageMetricsAdapter;
import org.apache.ignite.Ignite;
import org.apache.ignite.IgniteAtomicLong;
import org.apache.ignite.IgniteAtomicReference;
import org.apache.ignite.IgniteAtomicSequence;
import org.apache.ignite.IgniteAtomicStamped;
import org.apache.ignite.IgniteBinary;
import org.apache.ignite.IgniteCache;
import org.apache.ignite.IgniteCheckedException;
import org.apache.ignite.IgniteCluster;
import org.apache.ignite.IgniteCompute;
import org.apache.ignite.IgniteCountDownLatch;
import org.apache.ignite.IgniteDataStreamer;
import org.apache.ignite.IgniteEvents;
import org.apache.ignite.IgniteException;
import org.apache.ignite.IgniteFileSystem;
import org.apache.ignite.IgniteLock;
import org.apache.ignite.IgniteLogger;
import org.apache.ignite.IgniteMessaging;
import org.apache.ignite.IgniteQueue;
import org.apache.ignite.IgniteScheduler;
import org.apache.ignite.IgniteSemaphore;
import org.apache.ignite.IgniteServices;
import org.apache.ignite.IgniteSet;
import org.apache.ignite.IgniteSpring;
import org.apache.ignite.IgniteState;
import org.apache.ignite.IgniteTransactions;
import org.apache.ignite.Ignition;
import org.apache.ignite.MemoryMetrics;
import org.apache.ignite.PersistenceMetrics;
import org.apache.ignite.cache.affinity.Affinity;
import org.apache.ignite.cluster.ClusterGroup;
import org.apache.ignite.configuration.AtomicConfiguration;
import org.apache.ignite.configuration.CacheConfiguration;
import org.apache.ignite.configuration.CollectionConfiguration;
import org.apache.ignite.configuration.IgniteConfiguration;
import org.apache.ignite.configuration.NearCacheConfiguration;
import org.apache.ignite.internal.util.typedef.G;
import org.apache.ignite.internal.util.typedef.internal.S;
import org.apache.ignite.lang.IgniteProductVersion;
import org.apache.ignite.plugin.IgnitePlugin;
import org.apache.ignite.plugin.PluginNotFoundException;
import org.jetbrains.annotations.Nullable;
import org.joda.time.Duration;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.event.ContextRefreshedEvent;

import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.worker.spring.DelayingWorkerServiceBeanSupport;

/**
 * Ignite Spring bean allows to bypass {@link Ignition} methods.
 * In other words, this bean class allows to inject new grid instance from
 * Spring configuration file directly without invoking static
 * {@link Ignition} methods. This class can be wired directly from
 * Spring and can be referenced from within other Spring beans.
 *
 * <p>
 * A note should be taken that Ignite instance is started after all other
 * Spring beans have been initialized and right before Spring context is refreshed.
 * That implies that it's not valid to reference IgniteSpringBean from
 * any kind of Spring bean init methods like {@link javax.annotation.PostConstruct}.
 * If it's required to reference IgniteSpringBean for other bean
 * initialization purposes, it should be done from a {@link ContextRefreshedEvent}
 * listener method declared in that bean.
 * </p>
 *
 * <p>
 * <h1 class="header">Spring Configuration Example</h1>
 * Here is a typical example of describing it in Spring file:
 * <pre name="code" class="xml">
 * &lt;bean id="mySpringBean" class="org.apache.ignite.IgniteSpringBean"&gt;
 *     &lt;property name="configuration"&gt;
 *         &lt;bean id="grid.cfg" class="org.apache.ignite.configuration.IgniteConfiguration"&gt;
 *             &lt;property name="igniteInstanceName" value="mySpringGrid"/&gt;
 *         &lt;/bean&gt;
 *     &lt;/property&gt;
 * &lt;/bean&gt;
 * </pre>
 * Or use default configuration:
 * <pre name="code" class="xml">
 * &lt;bean id="mySpringBean" class="org.apache.ignite.IgniteSpringBean"/&gt;
 * </pre>
 * <h1 class="header">Java Example</h1>
 * Here is how you may access this bean from code:
 * <pre name="code" class="java">
 * AbstractApplicationContext ctx = new FileSystemXmlApplicationContext("/path/to/spring/file");
 *
 * // Register Spring hook to destroy bean automatically.
 * ctx.registerShutdownHook();
 *
 * Ignite ignite = (Ignite)ctx.getBean("mySpringBean");
 * </pre>
 * <p>
 */
public class IgniteSpringBean extends DelayingWorkerServiceBeanSupport
        implements Ignite, DisposableBean, ApplicationContextAware, Externalizable
{
    private static final DynamicProperty<Boolean> igniteNodeAutoRestartEnabled =
            new DynamicProperty<>("streaming-ignite-node-auto-restart-enabled", true);

    private static final long serialVersionUID = 0L;

    private final ExponentialBackoff restartBackoff =
            new ExponentialBackoff(Duration.standardMinutes(1), 1.5)
                    .doNotResetOnSuccess();

    private final Supplier<IgniteConfiguration> cfgSupplier;

    private Ignite g;

    private volatile IgniteConfiguration cfg;

    private final Lock startLock = new ReentrantLock();

    private ApplicationContext appCtx;

    public IgniteSpringBean(Supplier<IgniteConfiguration> cfgSupplier) {
        this.cfgSupplier = cfgSupplier;
    }

    @Override
    public IgniteConfiguration configuration() {
        return cfg;
    }

    /**
     * Gets the spring application context this Ignite runs in.
     *
     * @return Application context this Ignite runs in.
     */
    public ApplicationContext getApplicationContext() throws BeansException {
        return appCtx;
    }

    @Override
    public void setApplicationContext(ApplicationContext ctx) throws BeansException {
        appCtx = ctx;
    }

    @Override
    public void destroy() {
        if (g != null) {
            // Do not cancel started tasks, wait for them.
            G.stop(g.name(), false);
        }
    }

    private void startIgnite() {
        cfg = cfgSupplier.get();

        try {
            g = IgniteSpring.start(cfg, appCtx);
        } catch (IgniteCheckedException e) {
            throw new IgniteException("Failed to start IgniteSpringBean", e);
        }
    }

    @Override
    public IgniteLogger log() {
        checkIgnite();

        return cfg.getGridLogger();
    }

    @Override
    public IgniteProductVersion version() {
        checkIgnite();

        return g.version();
    }

    @Override
    public IgniteCompute compute() {
        checkIgnite();

        return g.compute();
    }

    @Override
    public IgniteServices services() {
        checkIgnite();

        return g.services();
    }

    @Override
    public IgniteMessaging message() {
        checkIgnite();

        return g.message();
    }

    @Override
    public IgniteEvents events() {
        checkIgnite();

        return g.events();
    }

    @Override
    public ExecutorService executorService() {
        checkIgnite();

        return g.executorService();
    }

    @Override
    public IgniteCluster cluster() {
        checkIgnite();

        return g.cluster();
    }

    @Override
    public IgniteCompute compute(ClusterGroup grp) {
        checkIgnite();

        return g.compute(grp);
    }

    @Override
    public IgniteMessaging message(ClusterGroup prj) {
        checkIgnite();

        return g.message(prj);
    }

    @Override
    public IgniteEvents events(ClusterGroup grp) {
        checkIgnite();

        return g.events(grp);
    }

    @Override
    public IgniteServices services(ClusterGroup grp) {
        checkIgnite();

        return g.services(grp);
    }

    @Override
    public ExecutorService executorService(ClusterGroup grp) {
        checkIgnite();

        return g.executorService(grp);
    }

    @Override
    public IgniteScheduler scheduler() {
        checkIgnite();

        return g.scheduler();
    }

    @Override
    public String name() {
        checkIgnite();

        return g.name();
    }

    @Override
    public void resetLostPartitions(Collection<String> cacheNames) {
        checkIgnite();

        g.resetLostPartitions(cacheNames);
    }

    @Override
    public Collection<DataRegionMetrics> dataRegionMetrics() {
        checkIgnite();

        return g.dataRegionMetrics();
    }

    @Nullable
    @Override
    public DataRegionMetrics dataRegionMetrics(String memPlcName) {
        checkIgnite();

        return g.dataRegionMetrics(memPlcName);
    }

    @Override
    public DataStorageMetrics dataStorageMetrics() {
        checkIgnite();

        return g.dataStorageMetrics();
    }

    @Override
    public Collection<MemoryMetrics> memoryMetrics() {
        return DataRegionMetricsAdapter.collectionOf(dataRegionMetrics());
    }

    @Nullable
    @Override
    public MemoryMetrics memoryMetrics(String memPlcName) {
        return DataRegionMetricsAdapter.valueOf(dataRegionMetrics(memPlcName));
    }

    @Override
    public PersistenceMetrics persistentStoreMetrics() {
        return DataStorageMetricsAdapter.valueOf(dataStorageMetrics());
    }

    @Override
    public <K, V> IgniteCache<K, V> cache(@Nullable String name) {
        checkIgnite();

        return g.cache(name);
    }


    @Override
    public Collection<String> cacheNames() {
        checkIgnite();

        return g.cacheNames();
    }

    @Override
    public <K, V> IgniteCache<K, V> createCache(CacheConfiguration<K, V> cacheCfg) {
        checkIgnite();

        return g.createCache(cacheCfg);
    }

    @Override
    public <K, V> IgniteCache<K, V> getOrCreateCache(CacheConfiguration<K, V> cacheCfg) {
        checkIgnite();

        return g.getOrCreateCache(cacheCfg);
    }

    @Override
    public <K, V> IgniteCache<K, V> createCache(CacheConfiguration<K, V> cacheCfg,
            NearCacheConfiguration<K, V> nearCfg)
    {
        checkIgnite();

        return g.createCache(cacheCfg, nearCfg);
    }

    @Override
    public Collection<IgniteCache> createCaches(Collection<CacheConfiguration> cacheCfgs) {
        checkIgnite();

        return g.createCaches(cacheCfgs);
    }

    @Override
    public <K, V> IgniteCache<K, V> getOrCreateCache(CacheConfiguration<K, V> cacheCfg,
            NearCacheConfiguration<K, V> nearCfg) {
        checkIgnite();

        return g.getOrCreateCache(cacheCfg, nearCfg);
    }

    @Override
    public <K, V> IgniteCache<K, V> createNearCache(String cacheName, NearCacheConfiguration<K, V> nearCfg) {
        checkIgnite();

        return g.createNearCache(cacheName, nearCfg);
    }

    @Override
    public <K, V> IgniteCache<K, V> getOrCreateNearCache(@Nullable String cacheName,
            NearCacheConfiguration<K, V> nearCfg) {
        checkIgnite();

        return g.getOrCreateNearCache(cacheName, nearCfg);
    }

    @Override
    public <K, V> IgniteCache<K, V> getOrCreateCache(String cacheName) {
        checkIgnite();

        return g.getOrCreateCache(cacheName);
    }

    @Override
    public Collection<IgniteCache> getOrCreateCaches(Collection<CacheConfiguration> cacheCfgs) {
        checkIgnite();

        return g.getOrCreateCaches(cacheCfgs);
    }

    @Override
    public <K, V> IgniteCache<K, V> createCache(String cacheName) {
        checkIgnite();

        return g.createCache(cacheName);
    }

    @Override
    public <K, V> void addCacheConfiguration(CacheConfiguration<K, V> cacheCfg) {
        checkIgnite();

        g.addCacheConfiguration(cacheCfg);
    }

    @Override
    public void destroyCache(String cacheName) {
        checkIgnite();

        g.destroyCache(cacheName);
    }

    @Override
    public void destroyCaches(Collection<String> cacheNames) {
        checkIgnite();

        g.destroyCaches(cacheNames);
    }

    @Override
    public IgniteTransactions transactions() {
        checkIgnite();

        return g.transactions();
    }

    @Override
    public <K, V> IgniteDataStreamer<K, V> dataStreamer(@Nullable String cacheName) {
        checkIgnite();

        return g.dataStreamer(cacheName);
    }

    @Override
    public IgniteFileSystem fileSystem(String name) {
        checkIgnite();

        return g.fileSystem(name);
    }

    @Override
    public Collection<IgniteFileSystem> fileSystems() {
        checkIgnite();

        return g.fileSystems();
    }

    @Override
    public <T extends IgnitePlugin> T plugin(String name) throws PluginNotFoundException {
        checkIgnite();

        return g.plugin(name);
    }

    @Override
    public IgniteBinary binary() {
        checkIgnite();

        return g.binary();
    }

    @Override
    public void close() throws IgniteException {
        g.close();
    }

    @Nullable
    @Override
    public IgniteAtomicSequence atomicSequence(String name, long initVal, boolean create) {
        checkIgnite();

        return g.atomicSequence(name, initVal, create);
    }

    @Override
    public IgniteAtomicSequence atomicSequence(String name, AtomicConfiguration cfg, long initVal,
            boolean create) throws IgniteException
    {
        checkIgnite();

        return g.atomicSequence(name, cfg, initVal, create);
    }

    @Nullable
    @Override
    public IgniteAtomicLong atomicLong(String name, long initVal, boolean create) {
        checkIgnite();

        return g.atomicLong(name, initVal, create);
    }

    @Override
    public IgniteAtomicLong atomicLong(String name, AtomicConfiguration cfg, long initVal,
            boolean create) throws IgniteException
    {
        checkIgnite();

        return g.atomicLong(name, cfg, initVal, create);
    }

    @Nullable
    @Override
    public <T> IgniteAtomicReference<T> atomicReference(String name,
            @Nullable T initVal,
            boolean create)
    {
        checkIgnite();

        return g.atomicReference(name, initVal, create);
    }

    @Override
    public <T> IgniteAtomicReference<T> atomicReference(String name, AtomicConfiguration cfg,
            @Nullable T initVal, boolean create) throws IgniteException
    {
        checkIgnite();

        return g.atomicReference(name, cfg, initVal, create);
    }

    @Nullable
    @Override
    public <T, S> IgniteAtomicStamped<T, S> atomicStamped(String name,
            @Nullable T initVal,
            @Nullable S initStamp,
            boolean create)
    {
        checkIgnite();

        return g.atomicStamped(name, initVal, initStamp, create);
    }

    @Override
    public <T, S> IgniteAtomicStamped<T, S> atomicStamped(String name, AtomicConfiguration cfg,
            @Nullable T initVal, @Nullable S initStamp, boolean create) throws IgniteException
    {
        checkIgnite();

        return g.atomicStamped(name, cfg, initVal, initStamp, create);
    }

    @Nullable
    @Override
    public IgniteCountDownLatch countDownLatch(String name,
            int cnt,
            boolean autoDel,
            boolean create)
    {
        checkIgnite();

        return g.countDownLatch(name, cnt, autoDel, create);
    }

    @Nullable
    @Override
    public IgniteSemaphore semaphore(String name,
            int cnt,
            boolean failoverSafe,
            boolean create)
    {
        checkIgnite();

        return g.semaphore(name, cnt,
                failoverSafe, create);
    }

    @Nullable
    @Override
    public IgniteLock reentrantLock(String name,
            boolean failoverSafe,
            boolean fair,
            boolean create)
    {
        checkIgnite();

        return g.reentrantLock(name, failoverSafe, fair, create);
    }

    @Nullable
    @Override
    public <T> IgniteQueue<T> queue(String name,
            int cap,
            CollectionConfiguration cfg)
    {
        checkIgnite();

        return g.queue(name, cap, cfg);
    }

    @Nullable
    @Override
    public <T> IgniteSet<T> set(String name,
            CollectionConfiguration cfg)
    {
        checkIgnite();

        return g.set(name, cfg);
    }

    @Override
    public <K> Affinity<K> affinity(String cacheName) {
        return g.affinity(cacheName);
    }

    @Override
    public boolean active() {
        checkIgnite();

        //noinspection deprecation
        return g.active();
    }

    @Override
    public void active(boolean active) {
        checkIgnite();

        //noinspection deprecation
        g.active(active);
    }

    @Override
    public void writeExternal(ObjectOutput out) throws IOException {
        out.writeObject(g);
    }

    @Override
    public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException {
        g = (Ignite) in.readObject();

        cfg = g.configuration();
    }

    /**
     * Checks if this bean is valid.
     *
     * @throws IllegalStateException If bean is not valid, i.e. Ignite has already been stopped
     * or has not yet been started.
     */
    protected void checkIgnite() throws IllegalStateException {
        if (g == null) {
            throw new IllegalStateException("Ignite is in invalid state to perform this operation. " +
                    "It either not started yet or has already being or have stopped.\n" +
                    "Make sure that IgniteSpringBean is not referenced from any kind of Spring bean init methods " +
                    "like @PostConstruct}.\n" +
                    "[ignite=" + g + ", cfg=" + cfg + ']');
        }
    }

    @Override
    public String toString() {
        return S.toString(IgniteSpringBean.class, this);
    }

    @Override
    protected void execute() {
        restartWithBackoffIfNotStarted();
    }

    private void restartWithBackoffIfNotStarted() {
        if (!igniteNodeAutoRestartEnabled.get()) {
            return;
        }

        if (Ignition.state() == IgniteState.STARTED) {
            return;
        }

        runWithStartLockIfUnlocked(() -> restartBackoff.executeIfDelayIsOver(this::restart));
    }

    private void restart() {
        startIgnite();
    }

    private void runWithStartLockIfUnlocked(Runnable runnable) {
        boolean locked = startLock.tryLock();
        if (locked) {
            try {
                runnable.run();
            } finally {
                startLock.unlock();
            }
        }
    }
}
