package ru.yandex.chemodan.videostreaming.framework.m3u;

import lombok.EqualsAndHashCode;
import lombok.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.videostreaming.framework.media.units.BitRate;
import ru.yandex.misc.image.Dimension;
import ru.yandex.misc.lang.DefaultObject;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class ExtM3UMasterPlaylist {
    private final CollectionF<Item> items;

    public ExtM3UMasterPlaylist(CollectionF<Item> items) {
        this.items = items;
    }

    public String mkString() {
        return "#EXTM3U\n" +
                items.map(Item::mkString).mkString("");
    }

    public static abstract class Item extends DefaultObject {
        public abstract String mkString();
    }

    @Value
    @EqualsAndHashCode(callSuper = true)
    public static class SessionDataItem extends Item {
        String id;

        String value;

        public String mkString() {
            return "#EXT-X-SESSION-DATA:DATA-ID=\"" + id + "\",VALUE=\"" + value + "\"\n";
        }
    }

    @Value
    @EqualsAndHashCode(callSuper = true)
    public static class Stream extends Item {
        BitRate bandwidth;

        Dimension resolution;

        String uri;

        Option<String> audioNameO;

        public String mkString() {
            ListF<String> chunks = Cf.list(
                    "#EXT-X-STREAM-INF:PROGRAM-ID=1",
                    "BANDWIDTH=" + bandwidth.getBs(),
                    "RESOLUTION=" + resolution
            ).plus(audioNameO.map(this::getAudioChunks).getOrElse(Cf::list));
            return chunks.mkString(", ") + "\n"
                    + uri + "\n";
        }

        private ListF<String> getAudioChunks(String audioName) {
            return Cf.list(
                    "CODECS=\"avc1.64002a,mp4a.40.2\"",
                    "AUDIO=\"" + audioName + "\""
            );
        }
    }

    @Value
    @EqualsAndHashCode(callSuper = true)
    public static class Media extends Item {
        String groupId;

        String name;

        boolean useByDefault;

        boolean autoSelect;

        String uri;

        public String mkString() {
            return "#EXT-X-MEDIA:TYPE=AUDIO"
                    + ",GROUP-ID=\"" + groupId + "\""
                    + ",NAME=\"" + name + "\""
                    + ",DEFAULT=" + (useByDefault ? "YES" : "NO")
                    + ",AUTOSELECT=" + (autoSelect ? "YES" : "NO")
                    + ",URI=\"" + uri + "\""
                    + "\n";
        }
    }
}
