package ru.yandex.chemodan.videostreaming.framework.m3u;

import java.util.function.Consumer;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.misc.lang.DefaultObject;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class ExtM3UMediaPlaylist extends DefaultObject {
    private static final boolean DEFAULT_ALLOW_CACHE = true;

    private static final boolean DEFAULT_END_LIST = false;

    private final int version = 3;

    private final int mediaSequence = 0;

    private final boolean allowCache;

    private final Option<ExtM3UPlaylistType> type;

    private final int targetDuration;

    private final boolean endList;

    private final ListF<ExtM3UMediaSegment> mediaSegments;

    public ExtM3UMediaPlaylist(Number targetDuration) {
        this(DEFAULT_ALLOW_CACHE, Option.empty(), targetDuration, DEFAULT_END_LIST, Cf.list());
    }

    private ExtM3UMediaPlaylist(boolean allowCache, Option<ExtM3UPlaylistType> type, Number targetDuration,
            boolean endList, ListF<ExtM3UMediaSegment> mediaSegments) {
        this.allowCache = allowCache;
        this.type = type;
        this.targetDuration = Math.max(Math.round(targetDuration.floatValue()), 1);
        this.endList = endList;
        this.mediaSegments = mediaSegments;
    }

    public static ExtM3UMediaPlaylist consFromMediaSegments(Tuple2List<String, Number> urisWithDurations) {
        Number maxSegmentDuration = urisWithDurations.get2().max();
        return new ExtM3UMediaPlaylist(maxSegmentDuration)
                .withMediaSegments(ExtM3UMediaSegment.fromTuples(urisWithDurations));
    }

    public ExtM3UMediaPlaylist forbidCache() {
        return build(b -> b.allowCache = false);
    }

    public ExtM3UMediaPlaylist allowCache() {
        return build(b -> b.allowCache = true);
    }

    public ExtM3UMediaPlaylist withType(ExtM3UPlaylistType type) {
        return build(b -> b.type = Option.of(type));
    }

    public ExtM3UMediaPlaylist endList() {
        return build(b -> b.endList = true);
    }

    @SuppressWarnings("unused")
    public ExtM3UMediaPlaylist withMediaSegment(String uri, Number duration) {
        return withMediaSegment(new ExtM3UMediaSegment(uri, duration));
    }

    public ExtM3UMediaPlaylist withMediaSegment(ExtM3UMediaSegment mediaSegment) {
        return build(b -> b.mediaSegments = b.mediaSegments.plus1(mediaSegment));
    }

    public ExtM3UMediaPlaylist withMediaSegments(CollectionF<ExtM3UMediaSegment> mediaSegments) {
        return build(b -> b.mediaSegments = b.mediaSegments.plus(mediaSegments));
    }

    public String mkString() {
        return toRawPlaylist().mkString();
    }

    public ExtM3URawPlaylist toRawPlaylist() {
        return new ExtM3URawPlaylist.Builder()
                .addTag(ExtM3uTag.EXTM3U)
                .addTag(ExtM3uTag.EXT_X_VERSION, version)
                .addTag(ExtM3uTag.EXT_X_MEDIA_SEQUENCE, mediaSequence)

                .addTag(ExtM3uTag.EXT_X_ALLOW_CACHE, allowCache ? "YES" : "NO")
                .addTagO(ExtM3uTag.EXT_X_PLAYLIST_TYPE, type)

                .addTag(ExtM3uTag.EXT_X_TARGETDURATION, targetDuration)
                .addMediaSegments(mediaSegments)

                .addTagIf(ExtM3uTag.EXT_X_ENDLIST, endList)

                .build();
    }

    private ExtM3UMediaPlaylist build(Consumer<Builder> setF) {
        Builder builder = new Builder();
        setF.accept(builder);
        return builder.build();
    }

    private class Builder {
        boolean allowCache = ExtM3UMediaPlaylist.this.allowCache;

        Option<ExtM3UPlaylistType> type = ExtM3UMediaPlaylist.this.type;

        boolean endList = ExtM3UMediaPlaylist.this.endList;

        ListF<ExtM3UMediaSegment> mediaSegments = ExtM3UMediaPlaylist.this.mediaSegments;

        ExtM3UMediaPlaylist build() {
            return new ExtM3UMediaPlaylist(allowCache, type, targetDuration, endList, mediaSegments);
        }
    }
}
