package ru.yandex.chemodan.videostreaming.framework.m3u;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Collection;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.IteratorF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.misc.io.InputStreamReaderSource;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.ReaderSource;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;

/**
 * See http://tools.ietf.org/html/draft-pantos-http-live-streaming-11
 * @author Alexei Zakharov
 */
public class ExtM3URawPlaylist {
    private final ListF<Element> elements;

    private ExtM3URawPlaylist(Collection<Element> elements) {
        this.elements = Cf.toList(elements).unmodifiable();
    }

    public static ExtM3URawPlaylist parse(InputStreamSource inputStreamSource) {
        return new ExtM3URawPlaylist(parse(new InputStreamReaderSource(inputStreamSource)));
    }

    private static ListF<Element> parse(ReaderSource readerSource) {
        ListF<Element> playlist = Cf.arrayList();
        IteratorF<String> it = readerSource.readLines().iterator();
        for(Option<Element> elementO = parseElementO(it); elementO.isPresent(); elementO = parseElementO(it)) {
            playlist.add(elementO.get());
        }
        return playlist.unmodifiable();
    }


    private static Option<Element> parseElementO(IteratorF<String> iterator) {
        return parseElementO(iterator, Cf.list());
    }

    // TODO add support for multiline values for tags like:
    //    #EXT-X-MEDIA:TYPE=AUDIO,GROUP-ID="aac",NAME="Deutsch", \
    //       DEFAULT=NO,AUTOSELECT=YES,LANGUAGE="de", \
    //       URI="main/german-audio.m3u8"
    private static Option<Element> parseElementO(IteratorF<String> iterator, ListF<TagWithValue> uriTags) {
        if (!iterator.hasNext()) {
            return Option.empty();
        }

        String line = iterator.next().trim();
        if (line.startsWith("#EXT")) {
            TagWithValue tagWithValue = TagWithValue.parse(line);
            return tagWithValue.tag.isStandalone()
                    ? Option.of(tagWithValue)
                    : parseElementO(iterator, uriTags.plus1(tagWithValue));
        } else if (line.startsWith("#")) {
            return Option.of(Comment.parse(line));
        } else if (line.length() > 0) {
            return Option.of(new MediaSegment(line, uriTags));
        } else {
            return parseElementO(iterator);
        }
    }

    /**
     * Adds string prefix to all relative URIs in media segments
     */
    ExtM3URawPlaylist addUriPrefix(String prefix) {
        return new ExtM3URawPlaylist(
                elements.map(element -> element.addUriPrefix(prefix))
        );
    }

    public ExtM3URawPlaylist withSegmentUrlsAsQueryArgs(String url, String param) {
        return new ExtM3URawPlaylist(
                elements.map(element -> element.withSegmentUrlsAsQueryArgs(url, param))
        );
    }

    public String mkString() {
        StringBuilder sb = new StringBuilder();
        for (Element e : elements) {
            sb.append(e.mkString());
            sb.append("\n");
        }
        return sb.toString();
    }

    ListF<Element> getElements() {
        return elements;
    }

    public static class Builder {
        private ListF<Element> elements = Cf.arrayList();

        public Builder addTagIf(ExtM3uTag tag, boolean condition) {
            return condition ? addTag(tag) : this;
        }

        public Builder addTag(ExtM3uTag tag) {
            return addTag(new TagWithValue(tag));
        }

        public Builder addTag(ExtM3uTag tag, Object... value) {
            return addTag(TagWithValue.cons(tag, value));
        }

        public Builder addTagO(ExtM3uTag tag, Option<?> value) {
            return value.map(o -> addTag(tag, o))
                    .getOrElse(this);
        }

        public Builder addUri(String uri, ExtM3uTag tag, Object... values) {
            return addTag(
                    new MediaSegment(uri, TagWithValue.cons(tag, values))
            );
        }

        private Builder addTag(Element element) {
            elements.add(element);
            return this;
        }

        public Builder addMediaSegments(Collection<ExtM3UMediaSegment> mediaSegments) {
            for (ExtM3UMediaSegment mediaSegment : mediaSegments) {
                addUri(mediaSegment.uri, ExtM3uTag.EXTINF, mediaSegment.duration, mediaSegment.title);
            }
            return this;
        }

        public ExtM3URawPlaylist build() {
            return new ExtM3URawPlaylist(elements);
        }
    }

    interface Element {
        String mkString();

        default Element withSegmentUrlsAsQueryArgs(String url, String param) {
            return this;
        }

        default Element addUriPrefix(String prefix) {
            return this;
        }
    }

    static class TagWithValue extends DefaultObject implements Element {
        private final ExtM3uTag tag;

        private final ListF<String> values;

        TagWithValue(ExtM3uTag tag) {
            this(tag, Cf.list());
        }

        TagWithValue(ExtM3uTag tag, String... values) {
            this(tag, Cf.list(values));
        }

        private TagWithValue(ExtM3uTag tag, ListF<String> values) {
            this.tag = tag;
            this.values = values;
        }

        static TagWithValue cons(ExtM3uTag tag, Object... values) {
            return new TagWithValue(tag, Cf.list(values).map(Object::toString));
        }

        static TagWithValue parse(String line) {
            line = line.trim();
            Validate.isTrue(line.startsWith("#"));

            int colonIdx = line.indexOf(':');
            if (colonIdx != -1) {
                return new TagWithValue(
                        ExtM3uTag.R.fromValue(line.substring(1, colonIdx).toUpperCase()),
                        StringUtils.notEmptyO(line.substring(colonIdx + 1))
                                .map(s -> Cf.list(s.split(",", -1)))
                                .getOrElse(Cf.list())
                );
            } else {
                return new TagWithValue(
                        ExtM3uTag.R.fromValue(line.substring(1).toUpperCase())
                );
            }
        }

        public String mkString() {
            return "#" + tag.value() +
                    (values.isNotEmpty()
                            ? ":" + StringUtils.join(values, ",")
                            : ""
                    );
        }
    }

    static class MediaSegment extends DefaultObject implements Element {
        private final ListF<TagWithValue> tags;
        private final String uri;

        MediaSegment(String uri, TagWithValue... tags) {
            this(uri, Cf.list(tags));
        }

        MediaSegment(String uri, Collection<TagWithValue> tags) {
            this.tags = Cf.toList(tags);
            this.uri = uri;
        }

        @Override
        public MediaSegment addUriPrefix(String prefix) {
            return canBePrefixed()
                    ? withUri(prefix + uri)
                    : this;
        }

        private boolean canBePrefixed() {
            try {
                return !new URI(uri).isAbsolute();
            } catch (URISyntaxException e) {
                // malformed URI, let's leave it as it is
                return false;
            }
        }

        MediaSegment withUri(String newUri) {
            return new MediaSegment(newUri, tags);
        }

        @Override
        public MediaSegment withSegmentUrlsAsQueryArgs(String url, String param) {
            return withUri(UrlUtils.addParameter(url, param, uri));
        }

        public String mkString() {
            StringBuilder sb = new StringBuilder();
            for (TagWithValue tag : tags) {
                sb.append(tag.mkString());
                sb.append("\n");
            }
            sb.append(uri);
            return sb.toString();
        }

        ListF<TagWithValue> getTags() {
            return tags;
        }

        public String getUri() {
            return uri;
        }
    }

    static class Comment extends DefaultObject implements Element {
        private final String comment;

        Comment(String comment) {
            this.comment = comment;
        }

        static Comment parse(String line) {
            line = line.trim();
            Validate.isTrue(line.startsWith("#"));

            return new Comment(line.substring(1));
        }

        public String mkString() {
            return "#" + comment;
        }
    }
}
