package ru.yandex.chemodan.videostreaming.framework.m3u;

import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.InputStreamSourceUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.reflection.ClassX;
import ru.yandex.misc.test.Assert;

import static ru.yandex.chemodan.videostreaming.framework.m3u.ExtM3URawPlaylist.TagWithValue;


/**
 * @author Alexei Zakharov
 */
public class ExtM3URawPlaylistTest {
    @SuppressWarnings("unused")
    private static final Logger logger = LoggerFactory.getLogger(ExtM3URawPlaylistTest.class);

    @Test
    public void parse() {
        InputStreamSource playlistIs = InputStreamSourceUtils.wrap(getClass().getResourceAsStream("test.m3u8"));
        ExtM3URawPlaylist playlist = ExtM3URawPlaylist.parse(playlistIs);

        ExtM3URawPlaylist.MediaSegment yao11 = getSegmentWithUri(playlist, "yao11.ts");
        Assert.assertContains(yao11.getTags(), new TagWithValue(ExtM3uTag.EXTINF, "11", ""));

        ExtM3URawPlaylist.MediaSegment yao1 = getSegmentWithUri(playlist, "yao1.ts");
        Assert.assertContains(yao1.getTags(), new TagWithValue(ExtM3uTag.EXTINF, "10", ""));

        ListF<TagWithValue> standaloneTags = playlist.getElements()
                .filter(ClassX.wrap(TagWithValue.class).isInstanceF())
                .uncheckedCast();

        Assert.assertContains(standaloneTags, new TagWithValue(ExtM3uTag.EXT_X_VERSION, "3"));
        Assert.assertContains(standaloneTags, new TagWithValue(ExtM3uTag.EXT_X_TARGETDURATION, "16"));
        Assert.assertContains(standaloneTags, new TagWithValue(ExtM3uTag.EXT_X_MEDIA_SEQUENCE, "0"));

        Assert.assertContains(standaloneTags, new TagWithValue(ExtM3uTag.EXTM3U));
        Assert.assertContains(standaloneTags, new TagWithValue(ExtM3uTag.EXT_X_ENDLIST));
    }

    @Test
    public void mkString() {
        InputStreamSource playlistIs = InputStreamSourceUtils.wrap(getClass().getResourceAsStream("test.m3u8"));
        ExtM3URawPlaylist playlist = ExtM3URawPlaylist.parse(playlistIs);
        String strPlaylist = playlist.mkString();

        logger.info(strPlaylist);
        Assert.isTrue(strPlaylist.contains("#EXT-X-TARGETDURATION:16"));
        Assert.isTrue(strPlaylist.contains("#EXT-X-ENDLIST"));
        Assert.isTrue(strPlaylist.contains("#EXTINF:16,\nyao18.ts"));
        Assert.isTrue(strPlaylist.contains("#EXTINF:10,\nyao17.ts"));
        Assert.isTrue(strPlaylist.indexOf("yao17.ts") < strPlaylist.indexOf("yao18.ts"));
    }

    @Test
    public void segmentDurationsUsedAsIs() {
        Assert.equals("#EXTINF:5,\n"
                        + "0.ts\n"
                        + "#EXTINF:0.4,\n"
                        + "1.ts\n",
                new ExtM3URawPlaylist.Builder()
                        .addMediaSegments(
                                Cf.list(
                                        new ExtM3UMediaSegment("0.ts", 5),
                                        new ExtM3UMediaSegment("1.ts", 0.4)
                                )
                        )
                        .build()
                        .mkString()
        );
    }

    @Test
    public void addUriPrefix() {
        final String PREFIX = "http://yandex.ru/video-disk/";
        InputStreamSource playlistIs = InputStreamSourceUtils.wrap(getClass().getResourceAsStream("test.m3u8"));
        ExtM3URawPlaylist playlist = ExtM3URawPlaylist.parse(playlistIs);
        ExtM3URawPlaylist updatedPlaylist = playlist.addUriPrefix(PREFIX);

        ListF<ExtM3URawPlaylist.MediaSegment> segments = getSegments(playlist);
        ListF<ExtM3URawPlaylist.MediaSegment> updatedSegments = getSegments(updatedPlaylist);

        Assert.equals(segments.size(), updatedSegments.size());
        for (int i = 0; i < updatedSegments.size(); i++) {
            Assert.isTrue(updatedSegments.get(i).getUri().startsWith(PREFIX));
            Assert.isTrue(updatedSegments.get(i).getUri().endsWith(segments.get(i).getUri()));
        }
    }

    private static ExtM3URawPlaylist.MediaSegment getSegmentWithUri(ExtM3URawPlaylist playlist, String uri) {
        return getSegments(playlist)
                .filter(s -> uri.equals(s.getUri()))
                .firstO()
                .get();
    }

    private static ListF<ExtM3URawPlaylist.MediaSegment> getSegments(ExtM3URawPlaylist playlist) {
        return playlist.getElements()
                .filter(ClassX.wrap(ExtM3URawPlaylist.MediaSegment.class).isInstanceF())
                .uncheckedCast();
    }
}
