package ru.yandex.chemodan.videostreaming.framework.media;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.videostreaming.framework.ffmpeg.ffprobe.FFprobeFormat;
import ru.yandex.chemodan.videostreaming.framework.ffmpeg.ffprobe.FFprobeInfo;
import ru.yandex.chemodan.videostreaming.framework.ffmpeg.ffprobe.FFprobeStream;
import ru.yandex.chemodan.videostreaming.framework.media.units.AbstractFractionUnit;
import ru.yandex.chemodan.videostreaming.framework.media.units.AspectRatio;
import ru.yandex.chemodan.videostreaming.framework.media.units.BitRate;
import ru.yandex.chemodan.videostreaming.framework.media.units.Fraction;
import ru.yandex.chemodan.videostreaming.framework.media.units.FrameRate;
import ru.yandex.chemodan.videostreaming.framework.media.units.MediaTime;
import ru.yandex.chemodan.videostreaming.framework.media.units.Rotation;
import ru.yandex.chemodan.videostreaming.framework.media.units.SampleFrequency;
import ru.yandex.misc.image.Dimension;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class MediaInfo {
    private static final ListF<String> AUDIO_FORMATS_WITH_IMAGES = Cf.list("mp3", "flac");

    private static final int MIN_SUITABLE_PIXEL_AREA = 100;

    private static final BitRate SILENT_AUDIO_BIT_RATE = BitRate.fromKbps(3);

    private final Format format;

    private final ListF<VideoStream> videoStreams;

    private final ListF<AudioStream> audioStreams;

    private final String json;

    public MediaInfo(FFprobeInfo info, String json) {
        this.format = new Format(info.getFormat());
        this.videoStreams = filterAndSort(info.getVideoStreams().map(VideoStream::new));
        this.audioStreams = filterAndSort(info.getAudioStreams().map(AudioStream::new));
        this.json = json;
    }

    public static MediaInfo parse(String json) {
        return new MediaInfo(FFprobeInfo.parse(json), json);
    }

    public String getFormat() {
        return format.getName();
    }

    public Option<BitRate> getBitRate() {
        return format.getBitRate();
    }

    public Option<MediaTime> getDuration() {
        return Option.of(format.getDuration());
    }

    public boolean hasVideo() {
        return videoStreams.isNotEmpty();
    }

    public VideoStream getVideoStream() {
        return getVideoStreamO().get();
    }

    public Option<VideoStream> getVideoStreamO() {
        return videoStreams.firstO();
    }

    public Option<BitRate> getVideoBitRateO() {
        return getVideoStreamO().filterMap(VideoStream::getBitRate);
    }

    public Option<FrameRate> getFrameRateO() {
        return getVideoStreamO().filterMap(VideoStream::getFrameRate);
    }

    public Option<Dimension> getRotatedDimensionO() {
        return getDimensionO()
                .map(dim -> isRotatedLeftOrRight() ? dim.rotate() : dim);
    }

    private Option<Dimension> getDimensionO() {
        return getVideoStreamO()
                .filterMap(VideoStream::getDimension);
    }

    private boolean isRotatedLeftOrRight() {
        return getVideoStreamO()
                .filterMap(VideoStream::getRotation)
                .isMatch(r -> r.getCcwDegrees() % 180 != 0);
    }

    public boolean hasAudio() {
        return audioStreams.isNotEmpty();
    }

    public Option<AudioStream> getAudioStreamO() {
        return audioStreams.firstO();
    }

    public String getJson() {
        return json;
    }

    private static boolean isValidCodec(String name) {
        return StringUtils.isNotBlank(name) && !name.equalsIgnoreCase("unknown");
    }

    private static <T extends Stream<?>> ListF<T> filterAndSort(ListF<T> streams) {
        return streams.filter(Stream::isSuitable)
                .sortedByDesc(Stream::getPriority);
    }

    @SuppressWarnings("unused")
    public Option<FrameRate> getTranscodeFrameRate() {
        return getVideoStreamO().map(VideoStream::getTranscodeFrameRate);
    }

    public class Format extends DefaultObject {
        private final FFprobeFormat format;

        public Format(FFprobeFormat format) {
            this.format = format;
        }

        public boolean isAudio() {
            return AUDIO_FORMATS_WITH_IMAGES.containsTs(format.getName());
        }

        public String getName() {
            return format.getName();
        }

        public Option<BitRate> getBitRate() {
            return format.getBitRate();
        }

        public MediaTime getDuration() {
            return format.getDuration();
        }
    }

    public static abstract class Stream<T extends FFprobeStream> extends DefaultObject {
        protected final T stream;

        protected Stream(T stream) {
            this.stream = stream;
        }

        protected boolean isValid() {
            return stream.getCodecName().isMatch(MediaInfo::isValidCodec);
        }

        protected abstract float getPriority();

        protected boolean isSuitable() {
            return isValid();
        }

        public String getCodec() {
            return stream.getCodecName()
                    .getOrElse("");
        }

        public Option<BitRate> getBitRate() {
            return stream.getBitRate();
        }

        public String getId() {
            return String.valueOf(stream.getIndex());
        }

        public Option<MediaTime> getDuration() {
            return stream.getDuration();
        }
    }

    public class VideoStream extends Stream<FFprobeStream.Video> {
        public VideoStream(FFprobeStream.Video stream) {
            super(stream);
        }

        public Option<Dimension> getDimension() {
            return Option.of(stream.getDimension());
        }

        public FrameRate getTranscodeFrameRate() {
            return TranscodeFrameRateUtil.resolve(stream.getAvgFrameRate(), stream.getRFrameRate());
        }

        public Option<FrameRate> getFrameRate() {
            return Cf.list(stream.getAvgFrameRate(), stream.getRFrameRate())
                    .find(AbstractFractionUnit::isDefined);
        }

        private long getPixelArea() {
            return stream.getDimension().getSquare();
        }

        public FrameRate getAvgFrameRate() {
            return stream.getAvgFrameRate();
        }

        @Override
        protected float getPriority() {
            // Motion jpeg steams should have a lowest priority, use them only as last resort
            if (stream.getCodecName().map(String::toLowerCase).isSome("mjpeg")) {
                return Float.MIN_VALUE;
            }

            return getAvgFrameRate()
                    .toFraction()
                    .definedOr(Fraction.ONE)
                    .multiply(getPixelArea())
                    .toFloat();
        }

        /**
         * Some audio files might contain image streams, for example, album cover.
         * ffprobe sets such stream codec_type = "video", so don't count them as video files.
         */
        @Override
        protected boolean isSuitable() {
            return super.isSuitable() && !format.isAudio() && getPixelArea() > MIN_SUITABLE_PIXEL_AREA;
        }

        public Option<Fraction> getOrCalculateDisplayAspectRatio() {
            return stream.getDisplayAspectRatio()
                    .filter(AspectRatio::isValid)
                    .orElse(() -> Option.of(calcAspectRatioFromDimension()))
                    .map(AbstractFractionUnit::toFraction);
        }

        private AspectRatio calcAspectRatioFromDimension() {
            return AspectRatio.fromDimension(stream.getDimension());
        }

        public Option<Rotation> getRotation() {
            return stream.getTags().filterMap(FFprobeStream.Video.Tags::getRotate);
        }
    }

    public class AudioStream extends Stream<FFprobeStream.Audio> {
        public AudioStream(FFprobeStream.Audio stream) {
            super(stream);
        }

        protected float getPriority() {
            if (stream.getBitRate().isMatch(SILENT_AUDIO_BIT_RATE::ge)) {
                return Float.MIN_VALUE;
            }

            return isStereo() ? 1 : 0;
        }

        public boolean isStereo() {
            return stream.getChannelLayout().isMatch("stereo"::equalsIgnoreCase) || stream.getChannels() == 2;
        }

        public Option<Integer> getChannelsCount() {
            return Option.of(stream.getChannels());
        }

        public Option<SampleFrequency> getSampleFrequency() {
            return Option.of(stream.getSampleRate());
        }
    }
}
