package ru.yandex.chemodan.videostreaming.framework.media;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.videostreaming.framework.media.units.AbstractFractionUnit;
import ru.yandex.chemodan.videostreaming.framework.media.units.Fraction;
import ru.yandex.chemodan.videostreaming.framework.media.units.FrameRate;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class TranscodeFrameRateUtil {
    private static final FrameRate LOWER_SUITABLE_FRAME_RATE = new FrameRate(23);

    private static final FrameRate UPPER_SUITABLE_FRAME_RATE = new FrameRate(31);

    private static final Fraction CLOSEST_FRAME_RATES_FACTOR = new Fraction(1, 10);

    private static final FrameRate DEFAULT_FRAME_RATE = new FrameRate(25);

    private static final SetF<FrameRate> DEFAULT_FRAME_RATES =
            Cf.set(new FrameRate(25), new FrameRate(30));

    public static FrameRate resolve(FrameRate avgFrameRate, FrameRate rFrameRate) {
        return new Resolver(avgFrameRate, rFrameRate)
                .resolve();
    }

    private static boolean isInSuitableRange(FrameRate frameRate) {
        return frameRate.inRange(LOWER_SUITABLE_FRAME_RATE, UPPER_SUITABLE_FRAME_RATE);
    }

    private static Option<FrameRate> getIfInRangeElseWidespreadO(FrameRate frameRate) {
        return Option.when(isInSuitableRange(frameRate), frameRate)
                .orElse(() -> FrameRates.getWidespreadByDerived(frameRate));
    }

    private static FrameRate getClosestDefault(FrameRate frameRate) {
        return DEFAULT_FRAME_RATES.sortedBy(def -> calcProximity(frameRate, def))
                .first();
    }

    private static Fraction calcProximity(FrameRate frameRate, FrameRate defaultFrameRate) {
        Fraction factor = frameRate.divide(defaultFrameRate);
        return factor.minus(factor.round()).abs();
    }

    private static class Resolver {
        final FrameRate avgFrameRate;

        final FrameRate rFrameRate;

        Resolver(FrameRate avgFrameRate, FrameRate rFrameRate) {
            this.avgFrameRate = avgFrameRate;
            this.rFrameRate = rFrameRate;
        }

        FrameRate resolve() {
            Option<FrameRate> confident = getConfident();
            return confident.filterMap(TranscodeFrameRateUtil::getIfInRangeElseWidespreadO)
                    .orElse(this::getClosestDefault)
                    .getOrElse(DEFAULT_FRAME_RATE);
        }

        Option<FrameRate> getConfident() {
            return Option.when(isBothDefined() && hasClosestValues(), rFrameRate);
        }

        boolean isBothDefined() {
            return avgFrameRate.isDefined() && rFrameRate.isDefined();
        }

        boolean hasClosestValues() {
            return calcDiff().le(CLOSEST_FRAME_RATES_FACTOR);
        }

        Fraction calcDiff() {
            return avgFrameRate.toFraction()
                    .minus(rFrameRate.toFraction())
                    .abs();
        }

        Option<FrameRate> getClosestDefault() {
            return getConfident().orElse(getDefined())
                    .map(TranscodeFrameRateUtil::getClosestDefault);
        }

        Option<FrameRate> getDefined() {
            return Cf.list(avgFrameRate, rFrameRate)
                    .find(AbstractFractionUnit::isDefined);
        }
    }
}
