package ru.yandex.chemodan.videostreaming.framework.media.units;

import java.math.BigDecimal;
import java.util.function.Function;

import lombok.EqualsAndHashCode;
import lombok.Value;

import ru.yandex.chemodan.util.bender.JsonFieldLevelUnmarshaller;
import ru.yandex.misc.bender.serialize.BenderJsonWriter;
import ru.yandex.misc.bender.serialize.ToFieldMarshaller;
import ru.yandex.misc.bender.serialize.simpleType.SimpleTypeMarshallerSupport;

/**
 * @author Dmitriy Amelin (lemeh)
 */
@EqualsAndHashCode(callSuper = false)
public abstract class AbstractFractionUnit<T extends AbstractFractionUnit<T>> extends AbstractMediaUnit {
    protected final Fraction fraction;

    protected AbstractFractionUnit(Fraction fraction) {
        this.fraction = fraction;
    }

    protected abstract Helper<T> getHelper();

    public boolean isDefined() {
        return fraction.isDefined();
    }

    public boolean isZero() {
        return fraction.isZero();
    }

    public boolean inRange(T lower, T upper) {
        return fraction.inRange(lower.toFraction(), upper.toFraction());
    }

    public T toLowest() {
        return getHelper()
                .cons(fraction.toLowest());
    }

    public Fraction toFraction() {
        return fraction;
    }

    public BigDecimal toBigDecimal() {
        return fraction.toBigDecimal();
    }

    public String toSimpleString() {
        return fraction.toSimpleString(getHelper().getSeparator());
    }

    @Override
    public String toString() {
        return fraction.consToString(getClass(), getHelper().getSeparator());
    }

    public T multiply(long factor) {
        return getHelper().cons(fraction.multiply(factor));
    }

    public T divide(long factor) {
        return getHelper().cons(fraction.divide(factor));
    }

    public Fraction divide(T other) {
        return this.fraction.divide(other.fraction);
    }

    @Value
    protected static class Helper<T> {
        String separator;

        Function<Fraction, T> constructor;

        protected JsonFieldLevelUnmarshaller consUnmarshaller() {
            return BenderUtil.consStringUnmarshaller(this::parse);
        }

        private T parse(String value) {
            return constructor.apply(Fraction.parse(value));
        }

        public T cons(Fraction fraction) {
            return constructor.apply(fraction);
        }

        public ToFieldMarshaller consMarshaller() {
            return new SimpleTypeMarshallerSupport() {
                @Override
                protected String toStringValueForXml(Object o) {
                    return getStringValue(o);
                }

                @Override
                protected void writeJson(BenderJsonWriter json, Object o) {
                    json.writeString(getStringValue(o));
                }
            };
        }

        private String getStringValue(Object o) {
            return ((AbstractFractionUnit<?>) o).toSimpleString();
        }
    }
}
