package ru.yandex.chemodan.videostreaming.framework.media.units;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.MathContext;
import java.math.RoundingMode;

import lombok.RequiredArgsConstructor;
import lombok.Value;
import org.jetbrains.annotations.NotNull;

import ru.yandex.misc.lang.Validate;

@Value
@RequiredArgsConstructor
public class Fraction implements ExtendedComparable<Fraction> {
    public static final Fraction ZERO = new Fraction(BigInteger.ZERO, BigInteger.ONE);

    public static final Fraction ONE = new Fraction(BigInteger.ONE, BigInteger.ONE);

    BigInteger numerator;

    BigInteger denominator;

    public Fraction(long numerator, long denominator) {
        this(BigInteger.valueOf(numerator), BigInteger.valueOf(denominator));
    }

    public static Fraction parse(String value) {
        String[] chunks = value.split("[/:]", 2);
        return new Fraction(new BigInteger(chunks[0]), new BigInteger(chunks[1]));
    }

    public static Fraction lowest(BigInteger numerator, BigInteger denominator) {
        return new Fraction(numerator, denominator)
                .toLowest();
    }

    public Fraction abs() {
        return numerator.signum() < 0 || denominator.signum() < 0
                ? new Fraction(numerator.abs(), denominator.abs())
                : this;
    }

    public Fraction plus(Fraction other) {
        return lowest(
                this.numerator.multiply(other.denominator).add(other.numerator.multiply(this.denominator)),
                this.denominator.multiply(other.denominator)
        );
    }

    public Fraction minus(Fraction other) {
        return lowest(
                this.numerator.multiply(other.denominator).subtract(other.numerator.multiply(this.denominator)),
                this.denominator.multiply(other.denominator)
        );
    }

    public Fraction multiply(long value) {
        return multiply(BigInteger.valueOf(value));
    }

    public Fraction multiply(BigInteger value) {
        return new Fraction(numerator.multiply(value), denominator);
    }

    public Fraction multiply(Fraction other) {
        return new Fraction(numerator.multiply(other.numerator), denominator.multiply(other.denominator));
    }

    public Fraction divide(long factor) {
        return divide(BigInteger.valueOf(factor));
    }

    public Fraction divide(BigInteger factor) {
        return lowest(numerator, denominator.multiply(factor));
    }

    public Fraction divide(Fraction other) {
        return new Fraction(numerator.multiply(other.denominator), denominator.multiply(other.numerator));
    }

    private Fraction divideBoth(BigInteger divisor) {
        validateNoRemainder(numerator, divisor, "numerator");
        validateNoRemainder(denominator, divisor, "denominator");
        return new Fraction(numerator.divide(divisor), denominator.divide(divisor));
    }

    private static void validateNoRemainder(BigInteger dividend, BigInteger divisor, String name) {
        Validate.isTrue(dividend.remainder(divisor).equals(BigInteger.ZERO),
                String.format("Divisor MUST divide %s without remainder: %d %% %d != 0", name, dividend, divisor)
        );
    }

    public Fraction invertIfProper() {
        return isDefined() && isProper() ? invert() : this;
    }

    public Fraction invert() {
        return new Fraction(denominator, numerator);
    }

    public Fraction definedOr(Fraction other) {
        return isDefined() ? this : other;
    }

    public Fraction zeroIfNegative() {
        return isNegative() ? ZERO : this;
    }

    public boolean isProper() {
        return numerator.abs().compareTo(denominator.abs()) < 0;
    }

    public boolean isDefined() {
        return !denominator.equals(BigInteger.ZERO);
    }

    public boolean isZero() {
        return numerator.equals(BigInteger.ZERO) && !denominator.equals(BigInteger.ZERO);
    }

    public boolean isNegative() {
        return (numerator.signum() < 0 && denominator.signum() >= 0)
                || (numerator.signum() >= 0 && denominator.signum() < 0);
    }

    public boolean inRange(Fraction lower, Fraction upper) {
        return ge(lower) && le(upper);
    }

    public Fraction round() {
        return new Fraction(Math.round(toFloat()), 1);
    }

    public long floor() {
        return (long) Math.floor(doubleValue());
    }

    public long ceil() {
        return (long) Math.ceil(doubleValue());
    }

    public Fraction toLowest() {
        return denominator.compareTo(BigInteger.ONE) > 0 ? divideBoth(numerator.gcd(denominator)) : this;
    }

    public BigDecimal toBigDecimal() {
        if (denominator.equals(BigInteger.ZERO)) {
            return BigDecimal.ZERO;
        }

        return new BigDecimal(numerator, MathContext.UNLIMITED)
                .divide(new BigDecimal(denominator, MathContext.UNLIMITED), 3, RoundingMode.HALF_UP);
    }

    public double doubleValue() {
        return toBigDecimal().doubleValue();
    }

    public float toFloat() {
        return toBigDecimal().floatValue();
    }

    @Override
    public int compareTo(@NotNull Fraction o) {
        return toBigDecimal().compareTo(o.toBigDecimal());
    }

    @Override
    public String toString() {
        return consToString(getClass(), "/");
    }

    String consToString(Class<?> clazz, String separator) {
        return String.format("%s(%s)", clazz.getSimpleName(), toSimpleString(separator));
    }

    String toSimpleString(String separator) {
        return String.format("%d%s%d", numerator, separator, denominator);
    }
}
