package ru.yandex.chemodan.videostreaming.framework.media.units;

import java.math.BigDecimal;
import java.math.MathContext;

import org.junit.Assert;
import org.junit.Test;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class FractionTest {
    @Test
    public void parse() {
        Assert.assertEquals(
                new Fraction(35, 248),
                Fraction.parse("35:248")
        );
        Assert.assertEquals(
                new Fraction(-100500, 2001000),
                Fraction.parse("-100500:2001000")
        );
    }

    @Test
    public void abs() {
        Assert.assertEquals(
                new Fraction(10, 3),
                new Fraction(-10, 3).abs()
        );
        Assert.assertEquals(
                new Fraction(5, 8),
                new Fraction(5, 8).abs()
        );
        Assert.assertEquals(
                new Fraction(41, 31),
                new Fraction(-41, -31).abs()
        );
    }

    @Test
    public void isProper() {
        Assert.assertTrue(new Fraction(4, 11).isProper());
        Assert.assertTrue(new Fraction(-12, 13).isProper());

    }

    @Test
    public void isNotProper() {
        Assert.assertFalse(new Fraction(9, 8).isProper());
        Assert.assertFalse(new Fraction(-25, 14).isProper());
        Assert.assertFalse(new Fraction(23, 23).isProper());
        Assert.assertFalse(new Fraction(-101, 101).isProper());
    }

    @Test
    public void isDefined() {
        assertDefined(new Fraction(-2, 3));
        assertDefined(new Fraction(0, 10));
        assertDefined(new Fraction(8, 37));
    }

    @Test
    public void isUndefined() {
        assertUndefined(new Fraction(-13, 0));
        assertUndefined(new Fraction(0, 0));
        assertUndefined(new Fraction(6, 0));
    }

    @Test
    public void isZero() {
        Assert.assertTrue(new Fraction(0, 100).isZero());
        Assert.assertTrue(new Fraction(0, -20).isZero());
        Assert.assertFalse(new Fraction(1, 2).isZero());
        Assert.assertFalse(new Fraction(0, 0).isZero());
        Assert.assertFalse(new Fraction(1, 0).isZero());
    }

    @Test
    public void inRange() {
        Fraction lower = new Fraction(1, 4);
        Fraction upper = new Fraction(9, 7);

        Assert.assertTrue(new Fraction(1, 4).inRange(lower, upper));
        Assert.assertTrue(new Fraction(1, 2).inRange(lower, upper));
        Assert.assertTrue(new Fraction(9, 7).inRange(lower, upper));

        Assert.assertFalse(new Fraction(1, 8).inRange(lower, upper));
        Assert.assertFalse(new Fraction(10, 7).inRange(lower, upper));
    }

    @Test
    public void round() {
        Assert.assertEquals(
                new Fraction(0, 1),
                new Fraction(1, 5).round()
        );
        Assert.assertEquals(
                new Fraction(2, 1),
                new Fraction(8, 5).round()
        );
    }

    @Test
    public void floor() {
        Assert.assertEquals(0, new Fraction(1, 5).floor());
        Assert.assertEquals(1, new Fraction(8, 5).floor());
    }

    @Test
    public void invert() {
        Assert.assertEquals(
                new Fraction(11, 43),
                new Fraction(43, 11)
                        .invert()
        );
    }

    @Test
    public void doInvertIfProper() {
        Assert.assertEquals(
                new Fraction(5, 4),
                new Fraction(4, 5)
                        .invertIfProper()
        );
        Assert.assertEquals(
                new Fraction(-100, 29),
                new Fraction(29, -100)
                        .invertIfProper()
        );
        Assert.assertEquals(
                new Fraction(499, 499),
                new Fraction(499, 499)
                        .invertIfProper()
        );
    }

    @Test
    public void returnSelfInsteadOfInvertIfNotProper() {
        Assert.assertEquals(
                new Fraction(101, 7),
                new Fraction(101, 7)
                        .invertIfProper()
        );
        Assert.assertEquals(
                new Fraction(-99, 10),
                new Fraction(-99, 10)
                        .invertIfProper()
        );
    }

    @Test
    public void plus() {
        Assert.assertEquals(
                new Fraction(5, 6),
                new Fraction(2, 3).plus(new Fraction(1,6))
        );
        Assert.assertEquals(
                new Fraction(3, 2),
                new Fraction(7, 10).plus(new Fraction(4,5))
        );
    }

    @Test
    public void minus() {
        Assert.assertEquals(
                new Fraction(1, 2),
                new Fraction(2, 3).minus(new Fraction(1,6))
        );
    }

    @Test
    public void multiply() {
        Assert.assertEquals(
                new Fraction(3, 20),
                new Fraction(1, 5).multiply(new Fraction(3, 4))
        );
    }

    @Test
    public void multiplyByLong() {
        Assert.assertEquals(
                new Fraction(46, 24),
                new Fraction(23, 24)
                        .multiply(2)
        );
    }

    @Test
    public void divideByLong() {
        Assert.assertEquals(
                new Fraction(2, 7),
                new Fraction(12, 7).divide(6)
        );
        Assert.assertEquals(
                new Fraction(13, 15),
                new Fraction(13, 3).divide(5)
        );
    }

    @Test
    public void divide() {
        Assert.assertEquals(
                new Fraction(20, 72),
                new Fraction(5, 8)
                        .divide(new Fraction(9, 4))
        );
    }

    @Test
    public void definedOrReturnsThisIfDefined() {
        Assert.assertEquals(
                new Fraction(15, 49),
                new Fraction(15, 49)
                        .definedOr(new Fraction(17, 51))
        );
    }

    @Test
    public void definedOrReturnsOtherIfUndefined() {
        Assert.assertEquals(
                new Fraction(19, 32),
                new Fraction(0, 0)
                        .definedOr(new Fraction(19, 32))
        );
    }

    @Test
    public void toLowest() {
        Assert.assertEquals(
                new Fraction(2,1),
                new Fraction(10,5)
                        .toLowest()
        );
        Assert.assertEquals(
                new Fraction(7,5),
                new Fraction(7,5)
                        .toLowest()
        );
        Assert.assertEquals(
                new Fraction(0,1),
                new Fraction(0,5)
                        .toLowest()
        );
    }

    @Test
    public void toLowestForUndefinedReturnsSame() {
        Assert.assertEquals(
                new Fraction(0, 0),
                new Fraction(0, 0)
                        .toLowest()
        );
        Assert.assertEquals(
                new Fraction(100, 0),
                new Fraction(100, 0)
                        .toLowest()
        );
    }

    @Test
    public void toFloat() {
        Assert.assertEquals(
                1.0 / 5.0,
                new Fraction(2, 10)
                        .toFloat(),
                0.00000001
        );
    }

    @Test
    public void toBigDecimal() {
        Assert.assertEquals(
                new BigDecimal("0.600", new MathContext(3)),
                new Fraction(3, 5).toBigDecimal()
        );
    }

    @Test
    public void consToString() {
        Assert.assertEquals(
                "Object(3/7)",
                new Fraction(3, 7)
                        .consToString(Object.class, "/")
        );
    }

    @Test
    public void compareTo() {
        assertCompareTo(new Fraction(1, 5), new Fraction(2, 3), -1);
        assertCompareTo(new Fraction(2, 4), new Fraction(1, 2), 0);
        assertCompareTo(new Fraction(7, 8), new Fraction(5, 9), 1);
    }

    private static void assertUndefined(Fraction fraction) {
        Assert.assertFalse("expected fraction to be undefined", fraction.isDefined());
    }

    private static void assertDefined(Fraction fraction) {
        Assert.assertTrue("expected fraction to be defined", fraction.isDefined());
    }

    private static void assertCompareTo(Fraction a, Fraction b, int expected) {
        Assert.assertEquals(expected, a.compareTo(b));
    }
}
