package ru.yandex.chemodan.videostreaming.framework.media.units;

import javax.annotation.Nonnull;

import lombok.EqualsAndHashCode;
import lombok.Value;
import org.joda.time.Duration;
import org.joda.time.Period;

import ru.yandex.chemodan.util.bender.JsonFieldLevelUnmarshaller;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author Dmitriy Amelin (lemeh)
 */
@Value
@EqualsAndHashCode(callSuper = false)
@Bendable
public class MediaTime extends AbstractMediaUnit implements ExtendedComparable<MediaTime> {
    public static final JsonFieldLevelUnmarshaller unmarshaller = BenderUtil.consStringUnmarshaller(MediaTime::parse);

    public static final MediaTime ZERO = new MediaTime(0);

    private static final int MICROS_PER_PER_MILLISECOND = 1000;

    public static final long MICROS_PER_SECOND = (long) Math.pow(10, 6);

    private static final long MICROS_PER_MINUTE = MICROS_PER_SECOND * 60;

    long micros;

    public static MediaTime fromPeriod(Period period) {
        return fromDuration(period.toStandardDuration());
    }

    public static MediaTime fromDuration(Duration duration) {
        return millis(duration.getMillis());
    }

    public static MediaTime minutes(int minutes) {
        return new MediaTime(minutes * MICROS_PER_MINUTE);
    }

    public static MediaTime seconds(long seconds) {
        return new MediaTime(seconds * MICROS_PER_SECOND);
    }

    public static MediaTime millis(long millis) {
        return new MediaTime(millis * MICROS_PER_PER_MILLISECOND);
    }

    public static MediaTime parse(String value) {
        String[] chunks = value.split("\\.");
        String seconds = chunks[0];
        String micros = padOrTruncateTo(chunks[1], 6);
        return seconds(Long.parseLong(seconds))
                .plus(Integer.parseInt(micros));
    }

    private static String padOrTruncateTo(String str, @SuppressWarnings("SameParameterValue") int length) {
        return str.length() > length
                ? StringUtils.substring(str, 0, length)
                : StringUtils.rightPad(str, length, '0');
    }

    public long getMillis() {
        return micros / 1000;
    }

    public MediaTime plus(long micros) {
        return new MediaTime(this.micros + micros);
    }

    public MediaTime plus(MediaTime other) {
        return new MediaTime(this.micros + other.micros);
    }

    public MediaTime minus(MediaTime other) {
        return new MediaTime(this.micros - other.micros);
    }

    public MediaTime multipliedBy(int multiplicand) {
        return new MediaTime(micros * multiplicand);
    }

    public MediaTime mod(MediaTime other) {
        return new MediaTime(micros % other.micros);
    }

    public boolean isLongerThan(MediaTime other) {
        return this.gt(other);
    }

    public boolean isShorterThan(MediaTime other) {
        return this.lt(other);
    }

    public Period toPeriod() {
        return toJodaDuration().toPeriod();
    }

    public Duration toJodaDuration() {
        return new Duration(micros / MICROS_PER_PER_MILLISECOND);
    }

    @Override
    public int compareTo(@Nonnull MediaTime other) {
        return Long.compare(this.micros, other.micros);
    }
}
