package ru.yandex.chemodan.videostreaming.framework.util;

import javax.servlet.http.HttpServletRequest;

import org.junit.Test;
import org.mockito.Mockito;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.misc.test.Assert;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class RequestAccessCheckerTest {
    private static final String EXTERNAL_HOST = "external.host.org";

    private static final String INTERNAL_HOST = "internal.host.org";

    private static final String EXTERNAL_PATH = "/hls/720p/playlist.m3u8";

    private static final String INTERNAL_PATH =
            "/info?source=320.yadisk:4003323001.E1614:297259517224825672142543749822";

    private static final RequestAccessChecker SUT = new RequestAccessChecker(Cf.set(INTERNAL_HOST), Cf.set("/hls", "/ping"));

    @Test
    public void mustDetectExternalHost() {
        Assert.isTrue(testIsFromExternal(EXTERNAL_HOST));
    }

    @Test
    public void mustDetectInternalHost() {
        Assert.isFalse(testIsFromExternal(INTERNAL_HOST));
    }

    @Test
    public void nullHostMustBeConsideredAsExternal() {
        Assert.isTrue(testIsFromExternal(null));
    }

    private static boolean testIsFromExternal(String host) {
        return SUT.isFromExternalHost(consRequest(host));
    }

    @Test
    public void requestFromExternalHostToInternalResourceIsForbidden() {
        Assert.isTrue(testIsForbidden(EXTERNAL_HOST, INTERNAL_PATH));
    }

    @Test
    public void requestFromExternalHostToExternalResourceIsNotForbidden() {
        Assert.isFalse(testIsForbidden(EXTERNAL_HOST, EXTERNAL_PATH));
    }

    @Test
    public void requestFromExternalHostToPingIsNotForbidden() {
        Assert.isFalse(testIsForbidden(EXTERNAL_HOST, "/ping"));
    }

    @Test
    public void requestFromInternalHostToInternalResourceIsNotForbidden() {
        Assert.isFalse(testIsForbidden(INTERNAL_HOST, INTERNAL_PATH));
    }

    @Test
    public void requestFromInternalHostToExternalResourceIsNotForbidden() {
        Assert.isFalse(testIsForbidden(INTERNAL_HOST, EXTERNAL_PATH));
    }

    private static boolean testIsForbidden(String host, String path) {
        HttpServletRequest request = consRequest(host);
        Mockito.when(request.getServletPath())
                .thenReturn(path);
        return SUT.isForbidden(request);
    }

    private static HttpServletRequest consRequest(String host) {
        HttpServletRequest request = Mockito.mock(HttpServletRequest.class);
        Mockito.when(request.getServerName())
                .thenReturn(host);
        return request;
    }
}
