package ru.yandex.chemodan.videostreaming.framework.util;

import java.math.BigDecimal;
import java.util.Objects;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsSegmentMeta;
import ru.yandex.chemodan.videostreaming.framework.media.units.MediaTime;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public interface StringParamConverter {
    SetF<String> TRUE_VALUES = Cf.set("true", "1");

    Option<String> getParameterO(String paramName);

    default Option<Integer> parseIntegerO(String paramName) {
        return getParameterO(paramName)
                .map(Integer::parseInt);
    }

    default Option<MediaTime> parseDurationFromDecimalSecondsO(@SuppressWarnings("SameParameterValue") String paramName) {
        return getParameterO(paramName)
                .map(durationStr -> MediaTime.millis(
                        new BigDecimal(durationStr)
                                .multiply(HlsSegmentMeta.MILLIS_PER_SECOND)
                                .longValue()
                ));
    }

    default Option<Duration> parseDurationFromMillisO(@SuppressWarnings("SameParameterValue") String paramName) {
        return getParameterO(paramName)
                .map(durationStr -> Duration.millis(Long.parseLong(durationStr)));
    }

    default Option<Double> parseDoubleO(@SuppressWarnings("SameParameterValue") String paramName) {
        return getParameterO(paramName)
                .map(Double::parseDouble);
    }

    default Boolean parseBoolean(String paramName, @SuppressWarnings("SameParameterValue") boolean defaultValue) {
        return parseBooleanO(paramName)
                .getOrElse(defaultValue);
    }

    default Option<Boolean> parseBooleanO(String paramName) {
        return getParameterO(paramName)
                .map(StringParamConverter::toBoolean);
    }

    static boolean toBoolean(String value) {
        return TRUE_VALUES.containsTs(value.toLowerCase());
    }

    class MapBased implements StringParamConverter {
        final MapF<String, String> params;

        public MapBased(MapF<String, String> params) {
            this.params = params.filterValues(Objects::nonNull);
        }

        @Override
        public Option<String> getParameterO(String paramName) {
            return params.getO(paramName);
        }
    }
}
