package ru.yandex.chemodan.videostreaming.framework.util.system;

import java.util.function.Function;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.lang.DefaultObject;

/**
 * @author Dmitriy Amelin (lemeh)
 *
 * http://man7.org/linux/man-pages/man5/proc.5.html
 * https://stackoverflow.com/questions/16726779/how-do-i-get-the-total-cpu-usage-of-an-application-from-proc-pid-stat/16736599#16736599
 */
public class LinuxProcFs {
    private static final ProcHelper uptimeHelper = new ProcHelper("/proc/uptime");

    private static final ProcHelper statHelper = new ProcHelper("/proc/stat");

    private static final ProcHelper selfStatHelper = new ProcHelper("/proc/self/stat");

    public static boolean isAvailable() {
        return statHelper.file.exists();
    }

    @SuppressWarnings("unused")
    public static Uptime readUptime() {
        return Uptime.read();
    }

    @SuppressWarnings("unused")
    public static CpuStat readCpuStat() {
        return CpuStat.read();
    }

    public static ProcessStat readSelfProcessStat() {
        return ProcessStat.read(selfStatHelper);
    }

    public static class Uptime extends DefaultObject {
        public final double uptime;

        public final double idleTime;

        private Uptime(double uptime, double idleTime) {
            this.uptime = uptime;
            this.idleTime = idleTime;
        }

        private static Uptime read() {
            return uptimeHelper.read(Uptime::parse);
        }

        private static Uptime parse(Data data) {
            return new Uptime(data.getDouble(0), data.getDouble(1));
        }
    }

    public static class CpuStat extends DefaultObject {
        public final long totalTime;

        private CpuStat(long totalTime) {
            this.totalTime = totalTime;
        }

        public static CpuStat read() {
            return statHelper.read(CpuStat::parse);
        }

        private static CpuStat parse(Data data) {
            return new CpuStat(data.getLongSum());
        }
    }

    public static class ProcessStat extends DefaultObject {
        public final long utime;
        public final long stime;
        public final long cutime;
        public final long cstime;
        public final long starttime;

        private ProcessStat(long utime, long stime, long cutime, long cstime, long starttime) {
            this.utime = utime;
            this.stime = stime;
            this.cutime = cutime;
            this.cstime = cstime;
            this.starttime = starttime;
        }

        public static ProcessStat read(ProcHelper procHelper) {
            return procHelper.read(ProcessStat::parse);
        }

        private static ProcessStat parse(Data data) {
            return new ProcessStat(
                    data.getLong(13),
                    data.getLong(14),
                    data.getLong(15),
                    data.getLong(16),
                    data.getLong(21)
            );
        }

        public long getTime() {
            return utime + stime;
        }

        public long getChildrenTime() {
            return cutime + cstime;
        }

        public long getTotalTime() {
            return getTime() + getChildrenTime();
        }
    }

    private static class ProcHelper extends DefaultObject {
        private final File2 file;

        public ProcHelper(String path) {
            this(new File2(path));
        }

        public ProcHelper(File2 file) {
            this.file = file;
        }

        public <T> T read(Function<Data, T> readerF) {
            return readerF.apply(readLine());
        }

        public Data readLine() {
            return new Data(file.readLine().split(" "));
        }
    }

    private static class Data extends DefaultObject {
        final String[] values;

        public Data(String[] values) {
            this.values = values;
        }

        public String getString(int index) {
            return values[index];
        }

        public long getLong(int index) {
            return Long.parseLong(getString(index));
        }

        public double getDouble(int index) {
            return Double.parseDouble(getString(index));
        }

        public long getLongSum() {
            return Cf.x(values).map(Long::parseLong).sum(Cf.Long);
        }
    }
}
