package ru.yandex.chemodan.videostreaming.framework.web;

import java.io.IOException;

import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.log.utils.ExtraRequestLogFieldsUtil;
import ru.yandex.chemodan.videostreaming.framework.hls.ShortFileInformation;
import ru.yandex.chemodan.videostreaming.framework.hls.StreamingUrlParams;
import ru.yandex.chemodan.videostreaming.framework.hls.stats.FFUtilStats;
import ru.yandex.chemodan.videostreaming.framework.util.system.AppCpuTime;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.web.servlet.HttpServletResponseX;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public abstract class AbstractHlsServlet extends HttpServlet {
    protected abstract void doGet(StreamingHttpServletRequest req, HttpServletResponseX resp) throws IOException;

    protected void doHead(
            @SuppressWarnings("unused") StreamingHttpServletRequest req,
            @SuppressWarnings("unused") HttpServletResponseX resp)
    {
        Validate.fail();
    }

    protected void doOptions(
            @SuppressWarnings("unused") StreamingHttpServletRequest req,
            @SuppressWarnings("unused") HttpServletResponseX resp)
    {
        Validate.fail();
    }

    @Override
    protected final void service(HttpServletRequest req, HttpServletResponse resp) {
        StreamingHttpServletRequest reqX = new StreamingHttpServletRequest(req);
        StreamingUrlParams.runWith(reqX.getUrlParams(),
                () -> FFUtilStats.runWithNew(
                    () -> service(reqX, HttpServletResponseX.wrap(resp))
                )
        );
    }

    private void service(StreamingHttpServletRequest reqX, HttpServletResponseX respX) {
        try {
            switch (reqX.getMethod()) {
                case "GET":
                    doGet(reqX, respX);
                    break;

                case "HEAD":
                    doHead(reqX, respX);
                    break;

                case "OPTIONS":
                    doOptions(reqX, respX);
                    break;

                default:
                    Validate.fail();
            }
        } catch (RuntimeException e) {
            if (e instanceof HlsErrorSource) {
                handleHlsError(respX, ((HlsErrorSource) e).getEffectiveHlsError());
            } else {
                throw e;
            }
        } catch (IOException e) {
            throw ExceptionUtils.translate(e);
        } finally {
            addStatsToAccessLog(reqX);
        }
    }

    private void handleHlsError(HttpServletResponseX respX, HlsErrorSource.HlsError hlsError) {
        try {
            respX.setStatus(hlsError.statusCode);
            respX.setContentType("text/plain; charset=UTF-8");
            try {
                respX.getOutputStream().write(hlsError.description.getBytes());
            } catch (IllegalStateException ex) {
                respX.getWriter().write(hlsError.description);
            }
        } catch (IOException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    private void addStatsToAccessLog(StreamingHttpServletRequest req) {
        FFUtilStats stats = FFUtilStats.getCurrent();
        MapF<String, String> logStats = Cf.hashMap();
        if (stats.getFileInformationO().isPresent()) {
            logStats.put("segment_src_meta",
                    ShortFileInformation.cons(stats.getFileInformationO().get())
                            .toJsonStr()
            );
        }
        addLoadStatToAccessLog("ffmpeg", stats.getFFmpegAppCpuTimeO(), logStats);
        addLoadStatToAccessLog("ffprobe", stats.getFFprobeAppCpuTimeO(), logStats);
        ExtraRequestLogFieldsUtil.addFields(req, logStats);
    }

    private void addLoadStatToAccessLog(String keyPrefix, Option<AppCpuTime.Interval> loadO,
            MapF<String, String> logStats)
    {
        if (loadO.isPresent()) {
            logStats.put(keyPrefix + "_time", String.valueOf(loadO.get().getDuration().getMillis()));
            if (loadO.get().getLoadO().isPresent()) {
                logStats.put(keyPrefix + "_app_load", loadO.get().getLoadO().get().toString());
            }
        }
    }
}
