package ru.yandex.chemodan.videostreaming.framework.web;

import java.nio.file.Path;
import java.nio.file.Paths;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.videostreaming.MpfsSourceMeta;
import ru.yandex.chemodan.videostreaming.framework.accesscheck.AccessChecker;
import ru.yandex.chemodan.videostreaming.framework.accesscheck.AccessParams;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsRequest;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsStreamQuality;
import ru.yandex.chemodan.videostreaming.framework.hls.sourcemeta.SourceMetaParser;
import ru.yandex.chemodan.videostreaming.framework.stidblocker.StidBlocker;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class HlsRequestParser<T> {
    private static final Logger logger = LoggerFactory.getLogger(HlsRequestParser.class);
    private static final DynamicProperty<Boolean> bannedIpUpdatesEnabled =
            DynamicProperty.cons("streaming-banned-ip-updates-enabled", true);
    private static final DynamicProperty<Boolean> checkBlockedStidsEnabled =
            new DynamicProperty<>("streaming-check-blocked-stids-enabled", true);

    private final SourceMetaParser<T> sourceMetaParser;

    private final AccessChecker accessChecker;

    private final BannedIpUpdater bannedIpUpdater;

    private final StidBlocker stidBlocker;

    public HlsRequestParser(SourceMetaParser<T> sourceMetaParser, AccessChecker accessChecker,
                            BannedIpUpdater bannedIpUpdater, StidBlocker stidBlocker)
    {
        this.sourceMetaParser = sourceMetaParser;
        this.accessChecker = accessChecker;
        this.bannedIpUpdater = bannedIpUpdater;
        this.stidBlocker = stidBlocker;
    }

    public HlsRequest<T> parse(StreamingHttpServletRequest request) {
        try {
            return parseUnsafe(request);
        } catch (RuntimeException e) {
            throw e instanceof HlsErrorSource ? e : new ParseException(e);
        }
    }

    private HlsRequest<T> parseUnsafe(StreamingHttpServletRequest request) {
        String uri = request.getPathInfo();
        ListF<String> segments = Cf.x(Paths.get(uri).iterator())
                .map(Path::toString)
                .map(UrlUtils::urlDecode)
                .toList();

        String filename = segments.last();
        Option<HlsStreamQuality> qualityO = tryParseQualityO(segments.get(segments.length() - 2));
        T srcMeta = sourceMetaParser.parse(request);
        HlsRequest<T> hlsRequest = new HlsRequest<>(filename, srcMeta, qualityO);
        checkAccess(AccessParams.fromRequests(request, hlsRequest));
        checkBannedStids(srcMeta);

        return hlsRequest;
    }

    private Option<String> getStidFromMeta(T srcMeta) {
        if (srcMeta instanceof MpfsSourceMeta) {
            MulcaId mulcaId = ((MpfsSourceMeta) srcMeta).getMulcaId();
            if (mulcaId != null) {
                return Option.ofNullable(mulcaId.getStid());
            }
        }

        return Option.empty();
    }

    private void checkBannedStids(T srcMeta) {
        getStidFromMeta(srcMeta).ifPresent(stid -> {
            if (checkBlockedStidsEnabled.get() && stidBlocker.isBlocked(stid)) {
                logger.info("Request by blocked stid");
                throw new SourceMetaParser.AccessForbiddenException(srcMeta);
            }
        });
    }

    private void checkAccess(AccessParams accessParams) {
        AccessChecker.Result checkResult = accessChecker.check(accessParams);
        if (bannedIpUpdatesEnabled.get()) {
            accessParams.onBothIpPresent((expectedIp, requestIp) -> {
                if (checkResult.isAllow()) {
                    return;
                }
                bannedIpUpdater.update(expectedIp, requestIp, accessParams.getSourceMeta());
            });
        }
    }

    private static Option<HlsStreamQuality> tryParseQualityO(String value) {
        try {
            return Option.of(HlsStreamQuality.parse(value));
        } catch (RuntimeException ex) {
            return Option.empty();
        }
    }

    private static class ParseException extends RuntimeException implements HlsErrorSource {
        public ParseException(RuntimeException cause) {
            super(cause);
        }

        @Override
        public HlsError getHlsError() {
            return new HlsError(HttpStatus.SC_400_BAD_REQUEST, "Could not parse request");
        }
    }
}
