package ru.yandex.chemodan.videostreaming.framework.web;

import java.io.InputStream;
import java.util.function.Consumer;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.log.utils.ExtraRequestLogFieldsUtil;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsManager;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsParams;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsParamsBySourceOverriderProvider;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsRequest;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsResponse;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsSegmentId;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsStreamQuality;
import ru.yandex.chemodan.videostreaming.framework.hls.segmentprovider.HlsStreamingManager;
import ru.yandex.chemodan.videostreaming.framework.hls.sourcemeta.SourceMetaConverter;
import ru.yandex.chemodan.videostreaming.framework.hls.sourcemeta.SourceMetaParser;
import ru.yandex.chemodan.videostreaming.framework.hls.sourcemeta.SourceMetaWithKey;
import ru.yandex.chemodan.videostreaming.framework.hls.sourcemeta.SourceMetaWithTranscodeParams;
import ru.yandex.chemodan.videostreaming.framework.hls.stats.HlsRequestStats;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.web.servlet.HttpServletResponseX;

/**
 * HTTP Live Streaming (HLS) servlet.
 *
 * <p>To be able to return relative segment uri in playlist, this servlet handles both playlists and segments.</p>
 *
 * <p>URI format: .../${serialized_resource_info}/${dimension}/${filename}</p>
 *
 * <p>Filename format: {"playlist.m3u8", "[0-9]+.ts"}</p>
 *
 * @author Dmitriy Amelin (lemeh)
 */
public class HlsServlet<T> extends AbstractHlsServlet {
    public static final String URL_PREFIX = "hls";

    private static final Logger logger = LoggerFactory.getLogger(HlsServlet.class);

    private static final DynamicProperty<Integer> fullHdPercentage =
            DynamicProperty.cons("streaming-full-hd-percentage", 0);

    public static final String AUDIO_SEGMENT_MEDIA_TYPE = "audio/aac";

    private final HlsRequestParser<T> requestParser;

    private final HlsManager<T> hlsManager;

    private final SourceMetaConverter<T> sourceMetaConverter;

    private final SetF<String> corsWhitelist;

    private final HlsParams hlsParams;

    private final HlsParamsBySourceOverriderProvider<T> hlsParamsBySourceOverriderProvider;

    public HlsServlet(
            HlsRequestParser<T> requestParser,
            HlsManager<T> hlsManager,
            SourceMetaConverter<T> sourceMetaConverter,
            SetF<String> corsWhitelist,
            HlsParams hlsParams,
            HlsParamsBySourceOverriderProvider<T> hlsParamsBySourceOverriderProvider)
    {
        this.requestParser = requestParser;
        this.hlsManager = hlsManager;
        this.sourceMetaConverter = sourceMetaConverter;
        this.corsWhitelist = corsWhitelist;
        this.hlsParams = hlsParams;
        this.hlsParamsBySourceOverriderProvider = hlsParamsBySourceOverriderProvider;
    }

    protected void doHead(StreamingHttpServletRequest req, HttpServletResponseX resp) {
        // do nothing like in old streaming
    }

    @Override
    protected void doOptions(StreamingHttpServletRequest req, HttpServletResponseX resp) {
        new HlsResponse(req, resp, corsWhitelist)
                .addCorsHeaders();
    }

    @Override
    protected void doGet(StreamingHttpServletRequest req, HttpServletResponseX resp) {
        logger.info("Request received: {}", req.getRequestUriWithQueryArgs());
        HlsRequestStats.runWithNew(() -> handleRequest(req, resp));
    }

    private void handleRequest(StreamingHttpServletRequest req, HttpServletResponseX resp) {
        Option<T> srcMetaO = Option.empty();
        try {
            HlsRequest<T> hlsReq = requestParser.parse(req);
            srcMetaO = Option.of(hlsReq.getSourceMeta());
            HlsResponse hlsResp = new HlsResponse(req, resp, corsWhitelist);
            boolean suppressOutput = req.parseBoolean("suppress_output", false);
            hlsParams.consOverridable()
                    .override(hlsParamsBySourceOverriderProvider.get(hlsReq.sourceMeta))
                    .override(new ParamOverrider(req.getParameterMap1()))
                    .override(o -> overrideExtra(o, hlsReq))
                    .run(() -> handleRequest(hlsReq, hlsResp, suppressOutput, req.isExtM3uSessionDataEnabled()));
        } catch(RuntimeException e) {
            if (e instanceof SourceMetaParser.SourceMetaException) {
                //noinspection unchecked
                srcMetaO = Option.of((T) ((SourceMetaParser.SourceMetaException) e).sourceMeta);
            }
            throw e;
        } finally {
            addStatsToAccessLog(req, srcMetaO);
        }
    }

    private HlsParams.Overridable overrideExtra(HlsParams.Overridable overridable, HlsRequest<T> request) {
        overridable = overridable.withStreamingResourceParams(builder -> builder.enableFullHd(isEnableFullHd(request)));

        if (!(request.sourceMeta instanceof SourceMetaWithTranscodeParams)) {
            return overridable;
        }

        SourceMetaWithTranscodeParams sourceMeta = (SourceMetaWithTranscodeParams) request.sourceMeta;
        return overridable.withFFmpegCommandParams(builder -> sourceMeta.getArgFiller().ifPresent(builder::argFillerName))
                .withFFmpegParams(builder -> sourceMeta.getFfVer().ifPresent(builder::ffmpegVersion));
    }

    private boolean isEnableFullHd(HlsRequest<T> request) {
        if (request.getSourceMeta() instanceof SourceMetaWithKey) {
            SourceMetaWithKey sourceMeta = (SourceMetaWithKey) request.getSourceMeta();
            boolean enableFullHd = Math.abs(sourceMeta.getKey().hashCode() % 100) < fullHdPercentage.get();
            logger.info("FullHD {} for source = {}", enableFullHd ? "enabled" : "disabled",
                    sourceMeta.getKey());
            return enableFullHd;
        } else {
            return false;
        }
    }

    private void handleRequest(
            HlsRequest<T> req, HlsResponse resp,
            boolean suppressOutput,
            boolean extM3uSessionDataEnabled
    ) {
        switch(req.filename) {
            case HlsManager.MASTER_PLAYLIST_FILENAME:
                resp.consumePlaylist(
                        hlsManager.buildMasterPlaylist(req.sourceMeta, extM3uSessionDataEnabled)
                                .mkString()
                );
                break;

            case HlsManager.PLAYLIST_FILENAME:
                resp.consumePlaylist(
                        hlsManager.buildPlaylist(req.sourceMeta, req.getQuality())
                                .toRawPlaylist()
                                .mkString()
                );
                break;

            default:
                if (!req.hasQuality()) {
                    resp.handleBadRequest();
                    return;
                }

                Option<HlsSegmentId> segmentIdO = HlsSegmentId.parseFilenameO(req.filename);
                if (!segmentIdO.isPresent()) {
                    resp.handleBadRequest();
                    return;
                }

                HlsSegmentId segmentId = segmentIdO.get();
                Consumer<InputStream> defaultConsumer = is -> resp.handleResponse(is, getMediaType(req));
                HlsManager<T>.SegmentResponseHandler respHandler =
                        hlsManager.consResponseHandler(req.sourceMeta, req.getQuality(), segmentId.index);
                switch (segmentId.type) {
                    case BASIC:
                        Consumer<InputStream> consumer = suppressOutput
                                ? HlsStreamingManager.devNullInputStreamConsumer
                                : defaultConsumer;
                        respHandler.stream(consumer);
                        if (suppressOutput) {
                            resp.handleResponse(
                                    DebugInfo.cons(req.getQuality())
                                            .toByteArrayInputStream(),
                                    "application/json"
                            );
                        }
                        break;

                    case PREFETCH:
                        respHandler.prefetch();
                        resp.handleOk();
                        break;

                    case LOCAL:
                        respHandler.streamLocalPrefetch(defaultConsumer);
                        break;

                    default:
                        resp.handleBadRequest();
                        logger.error("Do not know how to process segment of type {}", segmentId.type);
                }
        }
    }

    private static String getMediaType(HlsRequest<?> req) {
        return req.getQuality() == HlsStreamQuality._AAC ? AUDIO_SEGMENT_MEDIA_TYPE : HlsManager.SEGMENT_MEDIA_TYPE;
    }

    private void addStatsToAccessLog(StreamingHttpServletRequest req, Option<T> sourceMetaO) {
        HlsRequestStats stats = HlsRequestStats.getCurrent();
        MapF<String, String> logStats = Cf.hashMap();
        if (sourceMetaO.isPresent()) {
            logStats.put(sourceMetaConverter.getParamName(), sourceMetaConverter.convert(sourceMetaO.get()));
        }
        if (stats.getSourceO().isPresent()) {
            logStats.put("segment_data_src", stats.getSourceO().get().toString().toLowerCase());
        }
        if (stats.getDurationO().isPresent()) {
            logStats.put("segment_duration",
                    Long.valueOf(stats.getDurationO().get().getMillis())
                            .toString()
            );
        }
        ExtraRequestLogFieldsUtil.addFields(req, logStats);
    }
}
