package ru.yandex.chemodan.videostreaming.framework.web;

import org.joda.time.Duration;
import org.junit.Rule;
import org.junit.Test;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.junit.MockitoJUnit;
import org.mockito.junit.MockitoRule;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.chemodan.videostreaming.framework.accesscheck.BasicAccessChecker;
import ru.yandex.chemodan.videostreaming.framework.ffmpeg.FFmpegParams;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsManager;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsParams;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsParamsBySourceOverriderProvider;
import ru.yandex.chemodan.videostreaming.framework.hls.StreamingParams;
import ru.yandex.chemodan.videostreaming.framework.hls.ffmpeg.transcoding.HlsFFmpegCommandParams;
import ru.yandex.chemodan.videostreaming.framework.hls.sourcemeta.SourceMetaConverter;
import ru.yandex.chemodan.videostreaming.framework.hls.videoinfo.StreamingResourceParams;
import ru.yandex.chemodan.videostreaming.framework.hls.videoinfo.VideoInfoParams;
import ru.yandex.chemodan.videostreaming.framework.m3u.ExtM3UMasterPlaylist;
import ru.yandex.chemodan.videostreaming.framework.m3u.ExtM3UMediaPlaylist;
import ru.yandex.chemodan.videostreaming.framework.media.units.MediaTime;
import ru.yandex.chemodan.videostreaming.framework.stidblocker.InMemoryStidBlocker;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.web.servlet.mock.MockHttpServletRequest;
import ru.yandex.misc.web.servlet.mock.MockHttpServletResponse;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class HlsServletTest {
    @Rule
    public MockitoRule rule = MockitoJUnit.rule();

    @Mock
    private HlsManager<String>.SegmentResponseHandler segmentResponseHandler;

    @Mock
    private HlsManager<String> hlsManager;

    private MockHttpServletResponse response = new MockHttpServletResponse();

    @Test
    public void whenPathEndsWithQualityAndSegmentMustStream() {
        Mockito.when(hlsManager.consResponseHandler(Mockito.anyString(), Mockito.any(), Mockito.anyInt()))
                .thenReturn(segmentResponseHandler);
        requestHlsServlet("/stub/360p/1.ts");
        Mockito.verify(segmentResponseHandler)
                .stream(Mockito.any());
        Assert.equals(HttpStatus.SC_200_OK, response.getStatus());
    }

    @Test
    public void whenPathEndsWithQualityAndPlaylistMustBuildPlaylist() {
        Mockito.when(hlsManager.buildPlaylist(Mockito.anyString(), Mockito.any()))
                .thenReturn(new ExtM3UMediaPlaylist(5));
        requestHlsServlet("/stub/360p/playlist.m3u8");
        Mockito.verify(hlsManager)
                .buildPlaylist(Mockito.anyString(), Mockito.any());
        Assert.equals(HttpStatus.SC_200_OK, response.getStatus());
    }

    @Test
    public void whenPathEndsMasterPlaylistMustBuildMasterPlaylist() {
        Mockito.when(hlsManager.buildMasterPlaylist(Mockito.anyString(), Mockito.anyBoolean()))
                .thenReturn(new ExtM3UMasterPlaylist(Cf.list()));
        requestHlsServlet("/stub/master-playlist.m3u8");
        Mockito.verify(hlsManager)
                .buildMasterPlaylist(Mockito.anyString(), Mockito.anyBoolean());
        Assert.equals(HttpStatus.SC_200_OK, response.getStatus());
    }

    @Test
    public void whenPathIsUnknownMustReturn400() {
        requestHlsServlet("/stub/non-existing-file.txt");
        Assert.equals(HttpStatus.SC_400_BAD_REQUEST, response.getStatus());
    }

    private void requestHlsServlet(String pathInfo) {
        MockHttpServletRequest request = new MockHttpServletRequest("GET", "/hls" + pathInfo);
        request.setPathInfo(pathInfo);
        request.setQueryString("");
        consSut().service(request, response);
    }

    private HlsServlet<String> consSut() {
        return new HlsServlet<>(
                new HlsRequestParser<>(request -> "stub", BasicAccessChecker.alwaysNone(), BannedIpUpdater.noop(),
                        new InMemoryStidBlocker()),
                hlsManager,
                new SourceMetaConverter<>("param", s -> s),
                Cf.set(),
                consHlsParams(),
                HlsParamsBySourceOverriderProvider.noop()
        );
    }

    private static HlsParams consHlsParams() {
        return new HlsParams(
                new StreamingParams(MediaTime.seconds(5), 0, 3,
                        Duration.standardSeconds(20)),
                new VideoInfoParams(),
                FFmpegParams.DEFAULT,
                HlsFFmpegCommandParams.DEFAULT,
                StreamingResourceParams.DEFAULT
        );
    }
}
