package ru.yandex.chemodan.videostreaming.framework.web;

import java.util.Collections;
import java.util.Set;
import java.util.function.Supplier;

import lombok.Data;
import lombok.EqualsAndHashCode;
import org.apache.commons.lang3.RandomUtils;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.Option;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.misc.ip.IpAddress;
import ru.yandex.misc.ip.Network;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.worker.DelayingWorkerThread;

@Data
@EqualsAndHashCode(callSuper = true)
public class IpByNetworkMatcher extends DelayingWorkerThread implements IpMatcher {

    private final static Logger logger = LoggerFactory.getLogger(IpByNetworkMatcher.class);

    private final int maxRetries;
    private final Duration delayTime;

    private final Supplier<Set<Network>> networksSupplier;

    private volatile Set<Network> networks = Collections.emptySet();

    public IpByNetworkMatcher(Supplier<Set<Network>> networksSupplier) {
        this(networksSupplier, 3, Duration.standardMinutes(30));
    }

    public IpByNetworkMatcher(Supplier<Set<Network>> networksSupplier, int maxRetries, Duration delayTime) {
        super(IpByNetworkMatcher.class.getName());
        this.maxRetries = maxRetries;
        this.delayTime = delayTime;
        this.networksSupplier = networksSupplier;
        this.startGracefully();
    }

    @Override
    public boolean matches(IpAddress ip) {
        return networks.isEmpty() || networks.stream().anyMatch(m -> m.matches(ip));
    }

    public void update() {
        RetryUtils.retry(maxRetries, this::fetchNetworks)
                .filter(networks -> !networks.isEmpty()).ifPresent(n -> networks = n);
        logger.info("Fetched networks: {}", networks);
    }

    private Option<Set<Network>> fetchNetworks() {
        try {
            return Option.of(networksSupplier.get());
        } catch (Exception e) {
            logger.error("Failed to update vpns host", e);
            return Option.empty();
        }
    }

    @Override
    protected void executePeriodically() {
        update();
    }

    @Override
    protected long delayBetweenExecutionsMillis() {
        return delayTime.getMillis();
    }

    @Override
    protected Duration delayBeforeFirstRun() {
        return new Duration(RandomUtils.nextInt(0, 30_000));
    }

}
