package ru.yandex.chemodan.videostreaming.framework.web;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.videostreaming.framework.ffmpeg.FFmpegParams;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsManager;
import ru.yandex.chemodan.videostreaming.framework.hls.HlsParams;
import ru.yandex.chemodan.videostreaming.framework.hls.StreamingParams;
import ru.yandex.chemodan.videostreaming.framework.hls.ffmpeg.transcoding.HlsFFmpegCommandParams;
import ru.yandex.chemodan.videostreaming.framework.hls.videoinfo.VideoInfoParams;
import ru.yandex.chemodan.videostreaming.framework.media.units.MediaTime;
import ru.yandex.chemodan.videostreaming.framework.util.StringParamConverter;
import ru.yandex.misc.lang.DefaultObject;

/**
 * @author Dmitriy Amelin (lemeh)
 */
class ParamOverrider extends DefaultObject implements HlsParams.Overrider {
    private final StringParamConverter params;

    ParamOverrider(MapF<String, String> params) {
        this.params = new StringParamConverter.MapBased(params);
    }

    @Override
    public void override(HlsParams.Overridable overridable) {
        overridable
                .withStreamingParams(this::updateStreamingParams)
                .withVideoInfoParams(this::updateVideoInfoParams)
                .withFFmpegParams(this::updateFFmpegParams)
                .withFFmpegCommandParams(this::updateCommandParams);
    }

    private void updateStreamingParams(StreamingParams.StreamingParamsBuilder builder) {
        getSegmentDurationO()
                .ifPresent(builder::segmentDuration);
        getInitPrefetchCountO()
                .ifPresent(builder::initPrefetchSize);
        if (params.getParameterO(HlsManager.SEGMENT_CACHE_MISSING_PARAM).isSome("1")) {
            builder.fetchSegmentFromMaster(true);
        }
        builder.deleteCache(isDeleteCache())
                .disableDynamicPrefetch(isDisableDynamicPrefetch())
                .cacheDisabled(isCacheDisabled());
    }

    private Option<MediaTime> getSegmentDurationO() {
        return params.parseDurationFromDecimalSecondsO("segment_duration");
    }

    private Option<Integer> getInitPrefetchCountO() {
        return params.parseIntegerO("init_prefetch_size");
    }

    private boolean isDeleteCache() {
        return params.parseBoolean("delete_cache", false);
    }

    private boolean isDisableDynamicPrefetch() {
        return params.parseBoolean("disable_dynamic_prefetch", false);
    }

    private boolean isCacheDisabled() {
        return params.parseBoolean("disable_cache", false);
    }

    private void updateVideoInfoParams(VideoInfoParams.VideoInfoParamsBuilder builder) {
        builder.cacheDisabled(isCacheDisabled());
    }

    private void updateFFmpegParams(FFmpegParams.FFmpegParamsBuilder builder) {
        getLeftOverDurationO()
                .ifPresent(builder::minLeftOverDuration);
        getMaxProcCountO()
                .ifPresent(maxProcCount -> builder.maxProcCountO(Option.of(maxProcCount)));
        getSystemCpuLoadThresholdO()
                .ifPresent(cpuLoad -> builder.maxCpuLoadO(Option.of(cpuLoad)));
    }

    private Option<Duration> getLeftOverDurationO() {
        return params.parseDurationFromMillisO("left_over_duration");
    }

    private Option<Integer> getMaxProcCountO() {
        return params.parseIntegerO("max_proc_count");
    }

    private Option<Double> getSystemCpuLoadThresholdO() {
        return params.parseDoubleO("max_cpu_load");
    }

    private void updateCommandParams(HlsFFmpegCommandParams.HlsFFmpegCommandParamsBuilder builder) {
        getFFmpegInThreadsO()
                .ifPresent(builder::inputThreads);
        getFFmpegOutThreadsO()
                .ifPresent(builder::outputThreads);
    }

    private Option<Integer> getFFmpegInThreadsO() {
        return params.parseIntegerO("ffmpeg_in_threads");
    }

    private Option<Integer> getFFmpegOutThreadsO() {
        return params.parseIntegerO("ffmpeg_out_threads");
    }
}
