package ru.yandex.chemodan.app.webdav;

import java.security.NoSuchAlgorithmException;
import java.util.List;
import java.util.Optional;

import javax.net.ssl.SSLContext;

import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.chemodan.app.webdav.callback.MpfsCallbacksContextConfiguration;
import ru.yandex.chemodan.app.webdav.callback.MpfsCallbacksManager;
import ru.yandex.chemodan.app.webdav.filter.AuthenticationFilter;
import ru.yandex.chemodan.app.webdav.repository.MpfsResourceManager;
import ru.yandex.chemodan.app.webdav.repository.properties.DavPropertiesContextConfiguration;
import ru.yandex.chemodan.app.webdav.repository.properties.PropertiesFactory;
import ru.yandex.chemodan.app.webdav.repository.properties.PropertiesSetter;
import ru.yandex.chemodan.app.webdav.servlet.DavHandlersContextConfiguration;
import ru.yandex.chemodan.app.webdav.servlet.DavMethodHandler;
import ru.yandex.chemodan.app.webdav.servlet.JackrabbitHacks;
import ru.yandex.chemodan.app.webdav.servlet.NotFoundServlet;
import ru.yandex.chemodan.app.webdav.servlet.WebDavServlet;
import ru.yandex.chemodan.app.webdav.servlet.WebDavUserInitParamsExtractor;
import ru.yandex.chemodan.app.webdav.servlet.WebdavYcridPrefixResolver;
import ru.yandex.chemodan.blackbox.ProdBlackboxContextConfiguration;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsClientContextConfiguration;
import ru.yandex.chemodan.mpfs.MpfsUserInitParamsExtractor;
import ru.yandex.chemodan.mulca.MulcaClientContextConfiguration;
import ru.yandex.chemodan.util.ping.CoolPingServlet;
import ru.yandex.chemodan.util.web.A3JettyContextConfiguration;
import ru.yandex.chemodan.util.web.AddYcridToHeadersFilter;
import ru.yandex.chemodan.util.web.CustomJettyConfigurator;
import ru.yandex.chemodan.util.web.CustomYcridPrefixResolver;
import ru.yandex.chemodan.zk.registries.tvm.ZkTvm2MpfsContextConfiguration;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.misc.log.mlf.Level;
import ru.yandex.misc.web.servletContainer.jetty.MuteErrorHandler;

/**
 * @author tolmalev
 */
@Configuration
@Import({
        A3JettyContextConfiguration.class,
        MulcaClientContextConfiguration.class,
        MpfsCallbacksContextConfiguration.class,
        DavHandlersContextConfiguration.class,
        DavPropertiesContextConfiguration.class,
        ZkTvm2MpfsContextConfiguration.class,
        ProdBlackboxContextConfiguration.class
})
public class WebDavContextConfiguration {
    @Bean
    public CustomJettyConfigurator jettyConfigurator(
            AuthenticationFilter authenticationFilter,
            WebDavServlet webDavServlet,
            @Value("${a3.port}")
            int a3Port,
            @Value("${a3.proxy-protocol.port}")
            int proxyProtocolPort,
            @Value("${a3.sslSessionCacheSize:-}")
            Optional<Integer> sslSessionCacheSize,
            CoolPingServlet coolPingServlet)
            throws NoSuchAlgorithmException
    {
        if (sslSessionCacheSize.isPresent()) {
            SSLContext.getDefault().getClientSessionContext().setSessionCacheSize(sslSessionCacheSize.get());
        }
        JackrabbitHacks.configureXmlParserForSecurity();
        return jetty -> {
            jetty.setHttpPorts(Cf.set(a3Port));
            jetty.setProxyHttpPort(proxyProtocolPort);

            jetty.addServletMapping("/ping/*", coolPingServlet);

            jetty.addServletMapping("/*", webDavServlet);
            jetty.addServletMapping("/favicon.ico", new NotFoundServlet());

            jetty.addFilterMapping("*", new AddYcridToHeadersFilter());
            jetty.addFilterMapping("*", authenticationFilter);
            jetty.setFormEncodedMethods(Cf.set(HttpMethod.POST.name()));

            jetty.initialize();
            jetty.getContainerRootContext().setErrorHandler(new MuteErrorHandler(Level.DEBUG));
        };
    }

    @Bean
    public AuthenticationFilter oauthAuthenticationFilter(
            Blackbox2 blackbox2,
            @Value("${webdav.bad.karma.threshold}")
            int badKarmaThreshold,
            @Value("${webdav.oauth.trust.scopes}")
            String oauthTrustedScopes,
            @Value("${webdav.oauth.3rd.party.scopes}")
            String oauth3rdPartyScopes,
            AddAuthTypeInterceptor addAuthTypeInterceptor)
    {
        return new AuthenticationFilter(blackbox2, addAuthTypeInterceptor, badKarmaThreshold,
                Cf.x(oauthTrustedScopes.split(",")), Cf.x(oauth3rdPartyScopes.split(",")));
    }

    @MpfsClientContextConfiguration.Mpfs
    @Bean
    public AddAuthTypeInterceptor addAuthTypeInterceptor() {
        return new AddAuthTypeInterceptor();
    }

    @Bean
    public PropertiesSetter propertiesSetter(MpfsClient mpfsClient) {
        return new PropertiesSetter(mpfsClient);
    }

    @Bean
    public MpfsResourceManager diskDavResourceFactory(MpfsClient client,
            MpfsCallbacksManager callbacksManager,
            PropertiesSetter propertiesSetter,
            List<PropertiesFactory> propertiesFactories)
    {
        return new MpfsResourceManager(client, callbacksManager, propertiesSetter, Cf.x(propertiesFactories));
    }

    @Bean
    public WebDavServlet wedDavServlet(
            MpfsResourceManager mpfsResourceManager,
            List<DavMethodHandler> handlers)
    {
        return new WebDavServlet(mpfsResourceManager, Cf.x(handlers));
    }

    @Bean
    public CustomYcridPrefixResolver customYcridPrefixResolver() {
        return new WebdavYcridPrefixResolver();
    }

    @Bean
    public MpfsUserInitParamsExtractor userInitParamsExtractor() {
        return new WebDavUserInitParamsExtractor();
    }
}
