package ru.yandex.chemodan.app.webdav.auth;

import com.fasterxml.jackson.databind.JsonNode;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.util.json.JsonNodeUtils;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author tolmalev
 */
public class OurClient extends DefaultObject {
    private static final String YANDEX_DISK_UA_PREFIX = "Yandex.Disk";
    private static final Logger logger = LoggerFactory.getLogger(OurClient.class);

    public static final OurClient AUTOTESTS = new OurClient(
            "auto",
            Option.of("1.99999"),
            Option.of("autotests"),
            Option.of("autotests")
    );

    public final Type type;

    //windows, mac, macstore, ...
    public final String os;
    public final String platform;
    public final Option<String> version;
    public final Option<String> installId;
    public final Option<String> src;

    public final Option<Integer> build;
    public final Option<Integer> majorVersion;

    private boolean isMobMail;

    public OurClient(String os, Option<String> version, Option<String> installId, Option<String> src) {
        this.os = os;
        this.platform = os.startsWith("android") ? "android" : os;
        this.type = getType(platform);
        this.version = version;
        this.installId = installId;

        this.build = version.filterMap(OurClient::parseBuildFromVersion) ;
        this.src = src;
        this.isMobMail = assumeIsMobMail();
        this.majorVersion = version.filterMap(OurClient::parseMajorFromVersion);
    }

    private boolean assumeIsMobMail() {
        // don't know if we need it
        return ("android".equals(os) || "iOS".equals(os)) && !src.isSome("disk.mobile");
    }

    private Type getType(String platform) {
        switch (platform) {
            case "android":
            case "wp":
            case "iOS": return Type.MOBILE;

            case "windows":
            case "win8":
            case "winrt":
            case "mac":
            case "cli":
            case "macstore": return Type.DESKTOP;

            default: return Type.UNKNOWN;
        }
    }

    public static Option<OurClient> parseUserAgent(String ua) {
        if (ua == null) {
            return Option.empty();
        }
        ua = StringUtils.trim(ua);

        if (!ua.startsWith(YANDEX_DISK_UA_PREFIX)) {
            return Option.empty();
        }

        ua = StringUtils.removeStart(ua, YANDEX_DISK_UA_PREFIX);
        ua = StringUtils.trim(ua);

        ua = "{" + StringUtils.strip(ua, "({})") + "}";

        try {
            JsonNode node = JsonNodeUtils.getNode(ua);
            return Option.of(new OurClient(
                    node.get("os").asText(),
                    Option.ofNullable(node.get("vsn")).map(JsonNode::asText),
                    Option.ofNullable(node.get("id")).map(JsonNode::asText),
                    Option.ofNullable(node.get("src")).map(JsonNode::asText))
            );
        } catch (RuntimeException e) {
            logger.warn("Failed to parse User-Agent value to our client info : {}", e);
            return Option.empty();
        }
    }

    private static Option<Integer> parseBuildFromVersion(String version) {
        return Cf.list(version.split("\\.")).lastO()
                .flatMapO(Cf.Integer::parseSafe)
                .orElse(
                        Cf.list(version.split("-")).lastO().filterMap(Cf.Integer::parseSafe)
                );
    }

    private static Option<Integer> parseMajorFromVersion(String version) {
        return Cf.list(version.split("\\.")).firstO()
            .flatMapO(Cf.Integer::parseSafe)
            .orElse(
                Cf.list(version.split("-")).firstO().filterMap(Cf.Integer::parseSafe)
            );
    }

    public enum Type {
        DESKTOP,
        MOBILE,
        UNKNOWN
    }

}
