package ru.yandex.chemodan.app.webdav.callback;


import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.webdav.repository.MpfsCallback;
import ru.yandex.chemodan.http.YandexCloudRequestIdHolder;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.net.HostnameUtils;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.support.tl.ThreadLocalHandle;
import ru.yandex.misc.worker.AlarmThread;

/**
 * @author tolmalev
 */
public class MpfsCallbacksManager {
    private static final Logger logger = LoggerFactory.getLogger(MpfsCallbacksManager.class);

    private final int callbackJettyPort;

    private final Duration ourClientTimeout;
    private final Duration otherClientTimeout;

    private final TimeoutChecker timeoutChecker;
    private final MapF<String, RegisteredCallback> callbacks = Cf.concurrentHashMap();

    public MpfsCallbacksManager(int callbackJettyPort, Duration ourClientTimeout, Duration otherClientTimeout) {
        this.callbackJettyPort = callbackJettyPort;
        this.ourClientTimeout = ourClientTimeout;
        this.otherClientTimeout = otherClientTimeout;

        timeoutChecker = new TimeoutChecker();
        timeoutChecker.start();
    }

    public String registerCallback(boolean isOurClient, MpfsCallback callback) {
        RegisteredCallback registeredCallback =
                new RegisteredCallback(getTimeout(isOurClient), callback, Random2.R.nextAlnum(10));

        callbacks.put(registeredCallback.id, registeredCallback);

        return registeredCallback.id;
    }

    private Duration getTimeout(boolean isOurClient) {
        return isOurClient ? ourClientTimeout : otherClientTimeout;
    }

    public String getCallbackUrl(String callbackId) {
        return "http://" + HostnameUtils.localHostname() + ":" + callbackJettyPort + "/callback?id=" + callbackId;
    }

    public void callbackReceived(String callbackId, String requestBody) {
        MpfsOperationCallbackData callbackData = MpfsOperationCallbackData.PS.getParser().parseJson(requestBody);

        callbacks.getO(callbackId).forEach(registeredCallback -> {
            ThreadLocalHandle ycridHandle = YandexCloudRequestIdHolder.setAndPushToNdc(registeredCallback.ycrid);
            logger.debug("Callback received. callbackId={}, status={}. data={}",
                    callbackId, callbackData.getStatus(), requestBody
            );

            try {
                findAndRemove(callbackId).forEach(c -> c.mpfsCallback.onCallback(callbackData));
            } finally {
                ycridHandle.popSafely();
            }
        });
    }

    public Option<RegisteredCallback> findAndRemove(String callbackId) {
        return callbacks.removeO(callbackId);
    }

    private void checkTimeouts() {
        Instant now = Instant.now();

        ListF<String> idsToRemove = callbacks.values().filter(c -> c.deadline.isBefore(now)).map(c -> c.id);
        for (String id : idsToRemove) {
            ThreadLocalHandle ycridHandle = YandexCloudRequestIdHolder.setAndPushToNdc(callbacks.getTs(id).ycrid);
            logger.debug("Callback timeout. callbackId={}", id);
            try {
                findAndRemove(id).forEach(c -> c.mpfsCallback.onTimeout());
            } catch (Throwable e) {
                ExceptionUtils.throwIfUnrecoverable(e);
                logger.warn("Failed to call onTimeout for callbackId={}", id);
            } finally {
                ycridHandle.popSafely();
            }
        }
    }

    private static class RegisteredCallback {
        final Instant registerTime;
        final Instant deadline;
        final Duration timeout;

        final MpfsCallback mpfsCallback;
        final String id;
        final String ycrid;

        private RegisteredCallback(Duration timeout, MpfsCallback mpfsCallback, String id) {
            this.timeout = timeout;
            this.mpfsCallback = mpfsCallback;
            this.id = id;
            this.ycrid = YandexCloudRequestIdHolder.getO().getOrElse("not_set");

            registerTime = Instant.now();
            deadline = registerTime.plus(timeout);
        }
    }

    private class TimeoutChecker extends AlarmThread {
        TimeoutChecker() {
            super("MpfsCallbacks-timeout-checker", 1000);
        }

        @Override
        protected void alarm() {
            checkTimeouts();
        }
    }
}
