package ru.yandex.chemodan.app.webdav.filter;

import java.io.IOException;
import java.net.ConnectException;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;
import java.util.function.BiConsumer;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import net.jodah.failsafe.Failsafe;
import net.jodah.failsafe.FailsafeException;
import net.jodah.failsafe.Listeners;
import net.jodah.failsafe.RetryPolicy;
import net.jodah.failsafe.SyncFailsafe;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.webdav.AddAuthTypeInterceptor;
import ru.yandex.chemodan.app.webdav.auth.AuthInfo;
import ru.yandex.chemodan.app.webdav.auth.ClientCapabilities;
import ru.yandex.chemodan.app.webdav.auth.OurClient;
import ru.yandex.chemodan.app.webdav.log.WebdavApiTskvLogger;
import ru.yandex.chemodan.log.DiskLog4jRequestLog;
import ru.yandex.chemodan.log.utils.ExtraRequestLogFieldsUtil;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportDomain;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.inside.passport.blackbox2.protocol.BlackboxException;
import ru.yandex.inside.passport.blackbox2.protocol.BlackboxHttpException;
import ru.yandex.inside.passport.blackbox2.protocol.request.params.BlackboxAuthType;
import ru.yandex.inside.passport.blackbox2.protocol.request.params.BlackboxSid;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxCorrectResponse;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxFatalException;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxOAuthInfo;
import ru.yandex.inside.passport.blackbox2.protocol.response.Karma;
import ru.yandex.inside.passport.tvm2.TvmHeaders;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.codec.FastBase64Coder;
import ru.yandex.misc.io.RuntimeIoException;
import ru.yandex.misc.ip.IpAddress;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @author tolmalev
 */
public class AuthenticationFilter implements Filter {
    private static final Logger logger = LoggerFactory.getLogger(AuthenticationFilter.class);

    private final DynamicProperty<ListF<String>> trustedClients = new DynamicProperty<>("",
            Cf.list(
                    "f8cab64f154b4c8e96f92dac8becfcaa", // Поисковое приложение
                    "81ce3dcf5a054e1ebd73db3e28182a99", // Яндекс.Электрички
                    "2778917fcce1432fab2886c91b128ad0" // Yandex.Trains
            ));

    private static final String AUTH_HEADER = "Authorization";
    private static final String YANDEX_AUTH_HEADER = "Yandex-Authorization";
    private static final String BASIC_HEADER_START = "Basic ";
    private static final String OAUTH_HEADER_START = "OAuth ";
    private static final String BEARER_HEADER_START = "Bearer ";

    public static final String INFO_ATTR = "auth-info";

    private final SyncFailsafe blackboxFailsafe;

    private static final ListF<String> DBFIELDS = Cf.list(
            "subscription.login.81",
            "subscription.login.59",
            "subscription.login.669",
            "userinfo.lang.uid",
            "userinfo.country.uid",
            "userinfo.firstname.uid",
            "userinfo.lastname.uid"
    );

    private static final ListF<Integer> ATTRIBUTES = Cf.list(
            6, //account.is_pdd_agreement_accepted. "1", если подписан на sid=102
            1011
            //account.have_organization_name. Только для ПДД. "1" - если в опциях домена указано поле  organization_name
            , 191 // атрибут "Можно в webdav при помощи обычного пароля"
    );

    private final Blackbox2 blackbox;
    private final AddAuthTypeInterceptor addAuthTypeInterceptor;
    private final int badKarmaThreshold;
    private final ListF<String> oauthTrustedScopes;
    private final ListF<String> oauth3rdPartyScopes;

    public AuthenticationFilter(Blackbox2 blackbox, AddAuthTypeInterceptor addAuthTypeInterceptor,
            int badKarmaThreshold,
            ListF<String> oauthTrustedScopes, ListF<String> oauth3rdPartyScopes)
    {
        this.blackbox = blackbox;
        this.addAuthTypeInterceptor = addAuthTypeInterceptor;
        this.badKarmaThreshold = badKarmaThreshold;
        this.oauthTrustedScopes = oauthTrustedScopes;
        this.oauth3rdPartyScopes = oauth3rdPartyScopes;

        RetryPolicy policy = new RetryPolicy()
                .withMaxRetries(3)
                .withDelay(100, TimeUnit.MILLISECONDS)
                .retryOn(Cf.list(
                        ConnectException.class,
                        RuntimeIoException.class
                ));

        Listeners<?> listeners = new Listeners<>().onFailedAttempt((r, e, ctx) ->
                logger.warn("Failed to perform bb request#{}: {}", ctx.getExecutions(),
                        ExceptionUtils.getAllMessages(e)));

        blackboxFailsafe = Failsafe.with(policy).with(listeners);
    }

    public static AuthInfo getAuthInfo(ServletRequest req) {
        return (AuthInfo) req.getAttribute(INFO_ATTR);
    }

    private static void setAuthInfo(ServletRequest req, AuthInfo authInfo) {
        req.setAttribute(INFO_ATTR, authInfo);
        req.setAttribute(DiskLog4jRequestLog.AUTH_SOURCE_ATTRIBUTE, authInfo.authType.toString());
    }

    private static void setDeviceIdAndInstallId(ServletRequest req, AuthInfo authInfo) {
        Option<OurClient> ourClient = authInfo.ourClient;
        String installId = "-";
        String deviceId = "-";

        if (ourClient.isPresent()) {
            String id = ourClient.get().installId.getOrElse("-");
            switch (ourClient.get().type) {
                case MOBILE:
                    deviceId = id;
                    break;
                case DESKTOP:
                    installId = id;
                    break;
            }
        }
        ExtraRequestLogFieldsUtil.addFields(req, Cf.map("device_id", deviceId, "install_id", installId));
    }

    private static void setYandexUidResponseHeader(ServletResponse response, AuthInfo authInfo) {
        ((HttpServletResponse) response).addHeader("Yandex-Uid", authInfo.getUidStr());
    }

    @Override
    public void doFilter(ServletRequest req, ServletResponse response, FilterChain chain)
            throws IOException, ServletException
    {
        AuthInfo authInfo = extractAuthInfo((HttpServletRequest) req);

        setAuthInfo(req, authInfo);
        setDeviceIdAndInstallId(req, authInfo);
        setYandexUidResponseHeader(response, authInfo);

        try {
            addAuthTypeInterceptor.setAuthType(authInfo.authType);
            chain.doFilter(req, response);
        } finally {
            addAuthTypeInterceptor.clearAuthType();
        }
    }

    AuthInfo extractAuthInfo(HttpServletRequest req) {
        HttpServletRequestX reqX = HttpServletRequestX.wrap(req);

        IpAddress remoteIp = reqX.getRemoteIpAddress();

        Option<String> authHeaderO = reqX.getHeaderO(AUTH_HEADER).orElse(reqX.getHeaderO(YANDEX_AUTH_HEADER));

        return authHeaderO.flatMapO(
                authHeader -> tryOAuth(reqX, remoteIp, authHeader)
                        .orElse(() -> tryBasic(reqX, remoteIp, authHeader))
        ).getOrElse(() -> {
            Option<OurClient> ourClient = getOurClient(reqX);

            return new AuthInfo(ourClient, getClientCapabilities(ourClient, reqX));
        });
    }

    private static Option<OurClient> getOurClient(HttpServletRequestX reqX) {
        if (isAutotests(reqX)) {
            return Option.of(OurClient.AUTOTESTS);
        }
        return reqX.getUserAgent().flatMapO(OurClient::parseUserAgent);
    }

    private static boolean hasScopes(BlackboxOAuthInfo oAuthInfo, ListF<String> scopes) {
        return oAuthInfo.getScopes().count(scopes::containsTs) > 0;
    }

    private Option<AuthInfo> processPassportResponse(HttpServletRequestX req, BlackboxCorrectResponse resp,
            AuthInfo.AuthType authType)
    {
        PassportUid uid = resp.getUid().get();
        String login = resp.getLogin().get();

        if (!resp.getUidDomain().get()._2.sameAs(PassportDomain.YANDEX_RU)) {
            if (!resp.getAttributes().getO(6).isSome("1")) {
                return Option.empty();
            }
        }

        req.setAttribute(DiskLog4jRequestLog.UID_ATTRIBUTE, uid.toString());
        setLoginAttribute(req, login);

        Option<String> clientId = Option.empty();

        Option<Boolean> isPrivileged = Option.empty();

        if (resp.getOAuthInfo().isPresent()) {
            BlackboxOAuthInfo oAuthInfo = resp.getOAuthInfo().get();

            clientId = Option.of(oAuthInfo.getClientId());
            req.setAttribute(DiskLog4jRequestLog.CLIENT_ID_ATTRIBUTE, clientId.get());
            req.setAttribute(WebdavApiTskvLogger.CLIENT_INFO_ATTRIBUTE, oAuthInfo);

            isPrivileged = Option.of(hasScopes(oAuthInfo, oauthTrustedScopes));

            if (isPrivileged.isSome(false) && !hasScopes(oAuthInfo, oauth3rdPartyScopes)) {
                authType = AuthInfo.AuthType.BAD_SCOPE;
            }
        }

        Option<OurClient> ourClient = getOurClient(req);

        if (resp.getKarma().map(Karma::getValue).getOrElse(0) >= badKarmaThreshold) {
            logger.info("Client {} banned because of bad karma {}", clientId, resp.getKarma());
            authType = AuthInfo.AuthType.BANNED;
        }

        resp.getTvmUserTicket().forEach(ticket -> req.setAttribute(TvmHeaders.USER_TICKET, ticket));

        return Option.of(new AuthInfo(
                Option.of(uid),
                resp.getKarma(),
                Option.of(login),
                resp.getDbFields().getO("subscription.login.669").filter(StringUtils::isNotBlank),

                resp.getDbFields().getO("userinfo.firstname.uid").filter(StringUtils::isNotBlank),
                resp.getDbFields().getO("userinfo.lastname.uid").filter(StringUtils::isNotBlank),
                resp.getDbFields().getO("userinfo.country.uid").filter(StringUtils::isNotBlank),
                resp.getDbFields().getO("userinfo.lang.uid").filter(StringUtils::isNotBlank),

                resp.getAttributes().getO(1011).isSome("1"),

                clientId,

                ourClient,
                getClientCapabilities(ourClient, req),
                isPrivileged,
                resp.getAttributes().getO(191).filter(StringUtils::isNotBlank).map(v -> Objects.equals("1", v)),
                authType,
                !resp.getDbFields().getO("subscription.login.59").filter(StringUtils::isNotBlank).isPresent()
        ));
    }

    private void setLoginAttribute(HttpServletRequestX req, String login) {
        req.setAttribute(DiskLog4jRequestLog.LOGIN_ATTRIBUTE, login);
    }

    private ClientCapabilities getClientCapabilities(Option<OurClient> ourClient, HttpServletRequestX req) {
        // hack for out autotests
        if (isAutotests(req)) {
            return new ClientCapabilities(
                    false,
                    false,
                    false,
                    true,
                    "",
                    false
            );
        }
        ClientCapabilities capabilities = getForClient(ourClient);

        boolean allowChangeBaseLocation = ourClient.isPresent()
                || req.getUserAgent().map(ua -> ua.startsWith("ru.yandex.mail")).getOrElse(false)
                || trustedClients.get().containsTs(DiskLog4jRequestLog.getClientId(req));

        overrideFromHeader(capabilities, allowChangeBaseLocation, req.getHeaderO("Client-Capabilities"));
        return capabilities;
    }

    private static boolean isAutotests(HttpServletRequestX req) {
        return req.getUserAgent().isPresent() && req.getUserAgent().get().startsWith("Yandex.Disk Autotest");
    }

    private static final Map<String, BiConsumer<ClientCapabilities, Boolean>> CAPABILITIES = Cf
            .<String, BiConsumer<ClientCapabilities, Boolean>>map()
            .plus1("get_redirect", ClientCapabilities::setGetRedirect)
            .plus1("put_redirect", ClientCapabilities::setPutRedirect)
            .plus1("put_always_redirect", ClientCapabilities::setPutAlwaysRedirect)
            .plus1("use_autohide", ClientCapabilities::setUseAutohide)
            .plus1("delete_200", ClientCapabilities::setDelete200)
            .unmodifiable();

    private static void overrideFromHeader(ClientCapabilities capabilities, boolean allowChangeBaseLocation,
            Option<String> headerO)
    {
        headerO
                .flatMap(header -> Cf.list(header.split(",")))
                .map(StringUtils::trim)
                .forEach(cap -> {
                    if (CAPABILITIES.containsKey(cap)) {
                        CAPABILITIES.get(cap).accept(capabilities, true);
                    }
                    if (cap.startsWith("base_location=") && allowChangeBaseLocation) {
                        String base = cap.substring("base_location=".length());
                        if (base.equals("/")) {
                            base = "";
                        }
                        if (base.length() > 0 && !base.startsWith("/")) {
                            base = "/" + base;
                        }
                        capabilities.setBaseLocation(base);
                    }
                });
    }

    private static ClientCapabilities getForClient(Option<OurClient> ourClient) {
        return ourClient.map(client -> {
            //TODO: check capabilities for out clients
            ClientCapabilities capabilities = new ClientCapabilities(
                    true,
                    false,
                    false,
                    true,
                    "",
                    true
            );

            if (client.type == OurClient.Type.DESKTOP && !"win8".equals(client.os)) {
                capabilities.setPutRedirect(true);
            }
            if ("android".equals(client.os)) {
                capabilities.setUseAutohide(false);
            }
            return capabilities;
        }).getOrElse(ClientCapabilities.getDefault());
    }

    private Option<AuthInfo> tryOAuth(HttpServletRequestX req, IpAddress remoteIp, String authHeader) {
        Option<String> authToken = getAuthToken(authHeader, Cf.list(OAUTH_HEADER_START, BEARER_HEADER_START));

        return authToken.flatMapO(oAuthToken -> {
            try {
                return processPassportResponse(
                        req,
                        executeBbWithFailsafe(() -> blackbox.query().oAuth(
                                remoteIp, oAuthToken, DBFIELDS, ATTRIBUTES, Option.empty(), Option.empty(), true)),
                        AuthInfo.AuthType.OAUTH
                );
            } catch (BlackboxFatalException | BlackboxHttpException e) {
                logBBException(e);
                throw e;
            } catch (BlackboxException e) {
                logBBException(e);
                return Option.empty();
            }
        });
    }

    private Option<AuthInfo> tryBasic(HttpServletRequestX req, IpAddress remoteIp, String authHeader) {
        Option<String> authTokenO = getAuthToken(authHeader, Cf.list(BASIC_HEADER_START));
        return authTokenO.flatMapO(authToken -> {
            try {
                String decoded = new String(FastBase64Coder.decode(authToken));
                String[] split = decoded.split(":", 2);

                String login = split[0];
                String password = split[1];

                //if we receive response from, attribute will be overridden
                setLoginAttribute(req, login);

                return processPassportResponse(
                        req,
                        executeBbWithFailsafe(() -> blackbox.query().login(remoteIp, login, password,
                                Option.of(BlackboxSid.PASSPORT),
                                Option.empty(),
                                DBFIELDS,
                                ATTRIBUTES,
                                Option.of(BlackboxAuthType.WEBDAV),
                                true,
                                Option.of(true)
                        ).getOrThrow()),
                        AuthInfo.AuthType.BASIC
                );
            } catch (BlackboxFatalException | BlackboxHttpException e) {
                logBBException(e);
                throw e;
            } catch (RuntimeException e) {
                logBBException(e);
                return Option.empty();
            }
        });
    }

    private Option<String> getAuthToken(String authHeader, ListF<String> possibleStarts) {
        for (String start : possibleStarts) {
            if (authHeader.startsWith(start)) {
                return Option.of(authHeader.substring(start.length()));
            }
        }
        return Option.empty();
    }

    private <T> T executeBbWithFailsafe(Callable<T> callable) {
        try {
            return blackboxFailsafe.get(callable);
        } catch (FailsafeException e) {
            throw ExceptionUtils.throwException(e.getCause());
        }
    }

    @Override
    public void destroy() {
    }

    @Override
    public void init(FilterConfig filterConfig) {
    }

    private static void logBBException(RuntimeException exception) {
        logger.warn("Blackbox denial reason: {}", exception.getMessage());
        logger.warn("Exception in BB request: {}", exception);
    }
}
