package ru.yandex.chemodan.app.webdav.filter;

import org.apache.http.client.HttpClient;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.chemodan.app.webdav.AddAuthTypeInterceptor;
import ru.yandex.chemodan.app.webdav.auth.AuthInfo;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.inside.passport.blackbox2.BlackboxType;
import ru.yandex.inside.passport.tvm2.ByHostClientResolver;
import ru.yandex.inside.passport.tvm2.Tvm2;
import ru.yandex.inside.passport.tvm2.TvmClientCredentials;
import ru.yandex.misc.codec.FastBase64Coder;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.net.uri.Uri2;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.web.servlet.mock.MockHttpServletRequest;

/**
 * @author tolmalev
 */
public class AuthenticationFilterTest {
    private final TvmClientCredentials TEST_CREDENTIALS =
            new TvmClientCredentials(2000021, "CsENL4DVO-W-rueCSXIT-w");

    private final Tvm2 tvm2 = new Tvm2(TEST_CREDENTIALS);

    private final HttpClient httpClient = ApacheHttpClientUtils.Builder.create()
            .withHttpsSupport()
            .withInterceptorLast(tvm2.serviceTicketInterceptor(
                    new ByHostClientResolver(Cf.map(
                            Uri2.parse(BlackboxType.TEST.url()).getHost().get(),
                            BlackboxType.TEST.getTvmDstClientId()
                    ))))
            .build();

    private final AuthenticationFilter filter = new AuthenticationFilter(
            Blackbox2.cons(BlackboxType.TEST.url(), httpClient), new AddAuthTypeInterceptor(), 85, Cf.list("yadisk:all"), Cf.list("yadisk:disk"));

    @Before
    public void startTvm() {
        tvm2.setDstClientIds(Cf.list(BlackboxType.TEST.getTvmDstClientId()));
        tvm2.start();
    }

    @After
    public void stopTvm() {
        tvm2.stop();
    }

    @Test
    public void basicAuth() throws Exception {
        MockHttpServletRequest request = new MockHttpServletRequest();
        setAuthHeader(request, "Basic " + FastBase64Coder.encodeToString("testcontest00:asdfg12345".getBytes()));

        AuthInfo authInfo = filter.extractAuthInfo(request);
        Assert.isTrue(authInfo.isAuthorized());
        Assert.equals(AuthInfo.AuthType.BASIC, authInfo.authType);
    }

    @Test
    public void parseCapabilities() {
        MockHttpServletRequest request = new MockHttpServletRequest();
        setAuthHeader(request, "Basic " + FastBase64Coder.encodeToString("testcontest00:asdfg12345".getBytes()));
        request.addHeader("User-Agent", "Yandex.Disk (\"os\":\"wp\",\"vsn\":\"1.5.5968.45\",\"id\":\"96803A4621C9125CC596DA8F3853DE8837C98EE3\")");
        request.addHeader("Client-Capabilities", "put_always_redirect, base_location=/");

        AuthInfo authInfo = filter.extractAuthInfo(request);
        Assert.isTrue(authInfo.isAuthorized());
        Assert.equals(AuthInfo.AuthType.BASIC, authInfo.authType);

        Assert.isTrue(authInfo.clientCapabilities.isPutAlwaysRedirect());
        Assert.equals("", authInfo.clientCapabilities.getBaseLocation());
    }

    private void setAuthHeader(MockHttpServletRequest request, String value) {
        request.addHeader("Authorization", value);
    }
}
