package ru.yandex.chemodan.app.webdav.repository;

import java.io.IOException;
import java.util.concurrent.atomic.AtomicReference;

import javax.servlet.AsyncContext;
import javax.servlet.http.HttpServletResponse;

import lombok.AllArgsConstructor;
import lombok.RequiredArgsConstructor;
import org.apache.jackrabbit.webdav.MultiStatus;
import org.apache.jackrabbit.webdav.MultiStatusResponse;
import org.apache.jackrabbit.webdav.WebdavResponseImpl;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.function.Function3;
import ru.yandex.chemodan.app.webdav.auth.AuthInfo;
import ru.yandex.chemodan.app.webdav.callback.MpfsOperationCallbackData;
import ru.yandex.chemodan.app.webdav.filter.AuthenticationFilter;
import ru.yandex.chemodan.mpfs.MpfsOperation;
import ru.yandex.chemodan.util.exception.PermanentHttpFailureException;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author tolmalev
 */
@RequiredArgsConstructor
public class ComplexCallback implements MpfsCallback {
    private static final Logger logger = LoggerFactory.getLogger(ComplexCallback.class);

    private final MpfsResourceManager manager;

    private final AsyncContext ctx;
    private final ListF<AsyncOperation> operations;

    private final MultiStatus multiStatus = new MultiStatus();

    private int index = -1;

    @Override
    public synchronized void onCallback(MpfsOperationCallbackData callbackData) {
        AsyncOperation operation = operations.get(index);
        if (callbackData.isOk()) {
            sendOpStatus(operation, operation.okCode);
        } else {
            int errorResponseCode = callbackData.getErrorResponseCode();
            MultiStatusResponse singleStatus;

            if (HttpStatus.is5xx(errorResponseCode) && callbackData.getErrorTitleO().isPresent()) {
                singleStatus =
                        new MultiStatusResponse(operation.href, errorResponseCode, callbackData.getErrorTitleO().get());
            } else {
                singleStatus = new MultiStatusResponse(operation.href, errorResponseCode);
            }
            multiStatus.addResponse(singleStatus);
        }
        finishOrCallNextOperation();
    }

    @Override
    public synchronized void onTimeout() {
        AsyncOperation operation = operations.get(index);
        String oid = operation.operationId.get();
        if (oid == null) {
            sendOpStatus(operation, HttpStatus.SC_202_ACCEPTED);
        } else {
            //maybe we don't need it
            AuthInfo uid = AuthenticationFilter.getAuthInfo(ctx.getRequest());
            if (manager.getOperation(uid, oid).filter(o -> o.getStatus().isMatch("done"::equalsIgnoreCase)).isPresent()) {
                sendOpStatus(operation, operation.okCode);
            } else {
                sendOpStatus(operation, HttpStatus.SC_202_ACCEPTED);
            }
        }

        finishOrCallNextOperation();
    }

    private void sendOpStatus(AsyncOperation op, int status) {
        multiStatus.addResponse(new MultiStatusResponse(op.href, status));
    }

    public synchronized void finishOrCallNextOperation() {
        while (index <= operations.size()) {
            index++;
            if (index >= operations.size()) {
                // all operations finished
                // make response
                HttpServletResponse response = (HttpServletResponse) ctx.getResponse();

                try {
                    new WebdavResponseImpl(response).sendMultiStatus(multiStatus);
                } catch (IOException e) {
                    logger.error("Failed to send response: {}", e);
                } finally {
                    ctx.complete();
                }
                return;
            }
            AsyncOperation asyncOperation = operations.get(index);
            try {
                MpfsOperation operation = asyncOperation.fn.apply(ctx, asyncOperation.operationId, this);
                asyncOperation.operationId.set(operation.getOid());
                return;
            } catch (PermanentHttpFailureException e) {
                //operation failed
                sendOpStatus(asyncOperation, e.getStatusCode().getOrElse(500));
            } catch (HttpException e) {
                //operation failed
                sendOpStatus(asyncOperation, e.getStatusCode().getOrElse(500));
            }
        }
    }

    @AllArgsConstructor
    public static class AsyncOperation {

        public final String href;
        public final int okCode;
        public final Function3<AsyncContext, AtomicReference<String>, MpfsCallback, MpfsOperation> fn;

        final AtomicReference<String> operationId = new AtomicReference<>();

    }
}
