package ru.yandex.chemodan.app.webdav.repository;

import java.util.concurrent.atomic.AtomicReference;

import javax.servlet.AsyncContext;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import lombok.AllArgsConstructor;
import org.apache.jackrabbit.webdav.DavException;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.webdav.callback.MpfsOperationCallbackData;
import ru.yandex.chemodan.util.exception.PermanentHttpFailureException;
import ru.yandex.commune.a3.action.http.HttpStatusCodeSource;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.io.http.HttpStatus;

/**
 * @author tolmalev
 */
@AllArgsConstructor
public class MpfsCallbackBase implements MpfsCallback {

    private final AsyncContext ctx;
    private final AtomicReference<String> operationId;
    private final int okStatusCode;

    @Override
    public void onCallback(MpfsOperationCallbackData callbackData) {
        try {
            onCallbackWithException(callbackData);
        } catch (DavException e) {
            getResponse().setStatus(e.getErrorCode());
        } catch (PermanentHttpFailureException e) {
            getResponse().setStatus(e.getStatusCode().getOrElse(500));
        } catch (HttpException e) {
            getResponse().setStatus(e.getStatusCode().getOrElse(500));
        } catch (Throwable e) {
            ExceptionUtils.throwIfUnrecoverable(e);
            getResponse().setStatus(Option
                    .of(e)
                    .filter(HttpStatusCodeSource.class::isInstance)
                    .cast(HttpStatusCodeSource.class)
                    .map(HttpStatusCodeSource::getHttpStatusCode)
                    .getOrElse(HttpStatus.SC_500_INTERNAL_SERVER_ERROR)
            );
        } finally {
            ctx.complete();
        }
    }

    private void onCallbackWithException(MpfsOperationCallbackData callbackData) throws Exception {
        if (callbackData.isOk()) {
            getResponse().setStatus(okStatusCode);
        } else {
            if (HttpStatus.is5xx(callbackData.getErrorResponseCode())) {
                getResponse().setStatus(callbackData.getErrorResponseCode());
                callbackData.getErrorTitleO().ifPresent(getResponse().getWriter()::print);
            } else {
                getResponse().sendError(callbackData.getErrorResponseCode());
            }
        }
    }

    @Override
    public void onTimeout() {
        HttpServletResponse response = getResponse();

        response.setStatus(HttpStatus.SC_202_ACCEPTED);
        response.addHeader("X-Yandex-Op-ID", operationId.get());

        ctx.complete();
    }

    protected HttpServletResponse getResponse() {
        return (HttpServletResponse) ctx.getResponse();
    }

    protected HttpServletRequest getRequest() {
        return (HttpServletRequest) ctx.getRequest();
    }
}
