package ru.yandex.chemodan.app.webdav.repository;

import java.util.Collection;

import lombok.Getter;
import org.apache.jackrabbit.webdav.DavResource;
import org.apache.jackrabbit.webdav.DavResourceFactory;
import org.apache.jackrabbit.webdav.DavResourceLocator;
import org.apache.jackrabbit.webdav.property.DavPropertyName;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.app.webdav.auth.AuthInfo;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author tolmalev
 */
@Getter
public abstract class MpfsResourceBase extends AbstractMpfsDavResource {

    protected final AuthInfo user;
    protected final MpfsResourceManager manager;
    private final String realPath;
    protected final ListF<DavPropertyName> additionalPropertiesToFetch = Cf.arrayList();

    public MpfsResourceBase(AuthInfo user, MpfsResourceManager manager, String realPath,
            ListF<DavPropertyName> additionalPropertiesToFetch)
    {
        this.user = user;
        this.manager = manager;
        this.realPath = realPath;
        this.additionalPropertiesToFetch.addAll(additionalPropertiesToFetch);
    }

    public MpfsResourceBase(AuthInfo user, MpfsResourceManager manager, String realPath) {
        this(user, manager, realPath, Cf.list());
    }

    @Override
    public String getSupportedMethods() {
        return "OPTIONS, GET, HEAD, POST, PUT, DELETE, COPY, MOVE, MKCOL, PROPFIND, PROPPATCH, LOCK, UNLOCK, MKREDIRECTREF, UPDATEREDIRECTREF";
    }

    private String getBaseLocation() {
        return user.clientCapabilities.getBaseLocation();
    }

    private boolean isRoot() {
        return realPath.equals(getBaseLocation());
    }

    @Override
    public String getHref() {
        String href = StringUtils.removeStart(realPath, getBaseLocation());

        if ("/".equals(href)) {
            return href;
        }

        return PathUtils.encodePath(href) + (isCollection() ? "/" : "");
    }

    @Override
    public DavResource getCollection() {
        // I don't know when it is called but prefer to realize
        if (isRoot()) {
            return null;
        } else {
            String parentPath = Cf.list(StringUtils.split(realPath, "/")).rdrop(1).mkString("/");
            return manager.createResource(user, parentPath, additionalPropertiesToFetch, false);
        }
    }

    @Override
    public DavResourceLocator getLocator() {
        return new SimpleLocatorFactory("").createResourceLocator("", getResourcePath());
    }

    @Override
    public String getDisplayName() {
        return PathUtils.getDisplayName(realPath);
    }

    @Override
    public String getResourcePath() {
        return StringUtils.removeStart(realPath, getBaseLocation());
    }

    @Override
    public DavResourceFactory getFactory() {
        return manager;
    }

    @Override
    public void addFetchProperties(Collection<DavPropertyName> properties) {
        additionalPropertiesToFetch.addAll(properties);
    }
}
