package ru.yandex.chemodan.app.webdav.repository;

import java.util.Iterator;
import java.util.List;
import java.util.function.Supplier;

import org.apache.jackrabbit.webdav.DavException;
import org.apache.jackrabbit.webdav.DavResource;
import org.apache.jackrabbit.webdav.DavResourceFactory;
import org.apache.jackrabbit.webdav.DavResourceLocator;
import org.apache.jackrabbit.webdav.DavServletRequest;
import org.apache.jackrabbit.webdav.DavServletResponse;
import org.apache.jackrabbit.webdav.DavSession;
import org.apache.jackrabbit.webdav.MultiStatusResponse;
import org.apache.jackrabbit.webdav.property.DavProperty;
import org.apache.jackrabbit.webdav.property.DavPropertyName;
import org.apache.jackrabbit.webdav.property.DavPropertySet;
import org.apache.jackrabbit.webdav.property.PropEntry;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.chemodan.app.webdav.auth.AuthInfo;
import ru.yandex.chemodan.app.webdav.callback.MpfsCallbacksManager;
import ru.yandex.chemodan.app.webdav.filter.AuthenticationFilter;
import ru.yandex.chemodan.app.webdav.repository.properties.DavProperties;
import ru.yandex.chemodan.app.webdav.repository.properties.PropertiesFactory;
import ru.yandex.chemodan.app.webdav.repository.properties.PropertiesSetter;
import ru.yandex.chemodan.app.webdav.repository.upload.Upload;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.chemodan.mpfs.MpfsListResponse;
import ru.yandex.chemodan.mpfs.MpfsOperation;
import ru.yandex.chemodan.mpfs.MpfsShareFolderInvite;
import ru.yandex.chemodan.mpfs.MpfsStoreOperation;
import ru.yandex.chemodan.mpfs.MpfsStoreOperationContext;
import ru.yandex.chemodan.util.exception.PermanentHttpFailureException;
import ru.yandex.misc.io.http.HttpHeaderNames;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.ip.IpAddress;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @author tolmalev
 */
public class MpfsResourceManager implements DavResourceFactory {
    private static final Logger logger = LoggerFactory.getLogger(MpfsResourceManager.class);
    private static final ListF<DavPropertyName> WINDOWS_HACK_PROPERTY_NAMES = Cf.list(
            DavProperties.ISHIDDEN,
            DavProperties.HASSUBS,
            DavProperties.CHILDCOUNT,
            DavProperties.OBJECTCOUNT,
            DavProperties.MS_ATTR
    ).map(pd -> pd.name);

    private final MpfsClient mpfsClient;

    private final MpfsCallbacksManager callbacksManager;
    private final PropertiesSetter propertiesSetter;

    private final ListF<PropertiesFactory> propertiesFactories;

    public MpfsResourceManager(MpfsClient mpfsClient, MpfsCallbacksManager callbacksManager,
            PropertiesSetter propertiesSetter, ListF<PropertiesFactory> propertiesFactories)
    {
        this.mpfsClient = mpfsClient;
        this.callbacksManager = callbacksManager;
        this.propertiesSetter = propertiesSetter;
        this.propertiesFactories = propertiesFactories.sortedBy(PropertiesFactory::order);
    }

    public ListF<MpfsResource> createResources(
            AuthInfo authInfo, ListF<DavPropertyName> propertiesToRequest, ListF<String> fullPaths)
    {
        Option<SetF<String>> meta = buildMetaFields(propertiesToRequest).map(ListF::unique);
        return mpfsClient
                .bulkInfoByPaths(authInfo, meta.getOrElse(Cf.set()), fullPaths)
                .map(info -> toResource(authInfo, info, propertiesToRequest));
    }

    public MpfsRealResourceBase createResource(AuthInfo auth, String realPath,
            ListF<DavPropertyName> propertiesToRequest,
            boolean withChildren)
    {
        return createResource(auth, realPath, withChildren, Option.empty(), Option.empty(),
                Option.empty(), false, propertiesToRequest);
    }

    public MpfsRealResourceBase createResource(AuthInfo auth, String realPath, boolean withChildren,
            Option<Integer> offset, Option<Integer> amount,
            Option<String> sort, boolean orderDesc,
            ListF<DavPropertyName> propertiesToRequest)
    {
        Option<ListF<String>> meta = buildMetaFields(propertiesToRequest);

        if (withChildren) {
            MpfsListResponse response = mpfsClient.listByUidAndPath(auth, realPath, offset, amount, sort, orderDesc);

            if (isFile(response.rootInfo)) {
                return toResource(auth, response.rootInfo, propertiesToRequest);
            } else {
                return new DirectoryResource(auth, response.rootInfo, this,
                        Option.of(response.children.map(child -> toResource(auth, child, propertiesToRequest))),
                        propertiesToRequest
                );
            }
        } else {
            return toResource(auth, mpfsClient.getFileInfoByUidAndPath(auth, realPath, meta.getOrElse(Cf::list)), propertiesToRequest);
        }
    }

    private Option<ListF<String>> buildMetaFields(ListF<DavPropertyName> propertiesToSelect) {
        ListF<DavPropertyName> all = DavProperties.all
                .filter(DavProperties.PropertyDescription::isDefault)
                .filterNotNull()
                .map(DavProperties.PropertyDescription::getName);

        ListF<DavPropertyName> fullProperties = all.plus(propertiesToSelect);

        //check that all supported
        ListF<String> metaFields = Cf.arrayList();
        fullProperties.forEach(name -> propertiesFactories.filter(factory -> factory.accepts(name))
                .map(factory -> factory.getMetaFields(name)).forEach(metaFields::addAll));


        ListF<String> result = metaFields.stableUnique();
        logger.info("Build list of meta for mpfs: {}", result);

        return Option.of(result);
    }

    public void streamingListing(AuthInfo auth, String realPath,
            Option<Integer> offset, Option<Integer> amount,
            Option<String> sort, boolean orderDesc,
            ListF<DavPropertyName> propertiesToSelect,
            Function1V<Iterator<MpfsResource>> callback)
    {
        Option<ListF<String>> meta = buildMetaFields(propertiesToSelect);

        mpfsClient.streamingListByUidAndPath(auth, realPath, offset, amount, sort, orderDesc,
                (it) -> callback.apply(Cf.x(it).map(info -> toResource(auth, info, propertiesToSelect)))
        );
    }

    private MpfsRealResourceBase toResource(AuthInfo authInfo, MpfsFileInfo info,
            ListF<DavPropertyName> requestedProperties)
    {
        if (isFile(info)) {
            return new FileResource(authInfo, info, requestedProperties, this);
        } else {
            return new DirectoryResource(authInfo, info, this, Option.empty(), requestedProperties);
        }
    }

    private boolean isFile(MpfsFileInfo info) {
        return info.type.isSome("file");
    }

    @Override
    public DavResource createResource(DavResourceLocator locator, DavServletRequest request,
            DavServletResponse response) throws DavException
    {
        HttpServletRequestX reqX = HttpServletRequestX.wrap(request);
        AuthInfo authInfo = AuthenticationFilter.getAuthInfo(request);
        String path = extractPath(locator);

        if ((path.isEmpty() || path.equals("/"))) {
            String queryString = reqX.getQueryString() == null ? "" : reqX.getQueryString();

            if (reqX.getParameterO("hash").isPresent()) {
                return createPublicResource(authInfo, reqX.getParameter("hash"));
            } else if (StringUtils.removeEnd(queryString, "/").equals("share/not_approved")) {
                return new MpfsInvitesListResource(authInfo,
                        mpfsClient.getNotApprovedInvites(authInfo));
            } else if (queryString.startsWith("share/not_approved")) {
                String hash = StringUtils.removeStart(queryString, "share/not_approved/");
                MpfsShareFolderInvite invite = mpfsClient
                        .getNotApprovedInvite(authInfo, hash)
                        .getOrThrow(() -> new DavException(HttpStatus.SC_404_NOT_FOUND));

                return new MpfsInviteResource(authInfo, invite);
            }
        }

        boolean getChildren = request.getMethod().equals("PROPFIND") && "1".equals(request.getHeader("Depth"));

        String realPath;
        if (!path.startsWith("/share")) {
            realPath = authInfo.clientCapabilities.getBaseLocation() + path;
        } else {
            if (!authInfo.isAuthorized()) {
                realPath = path;
            } else {
                realPath = authInfo.clientCapabilities.getBaseLocation() + path;
            }

        }

        while (realPath.startsWith("//")) {
            realPath = StringUtils.removeStart(realPath, "/");
        }

        if (!realPath.startsWith("/")) {
            realPath = "/" + realPath;
        }

        String finalRealPath = realPath;
        return new MpfsUnloadedResource(realPath, authInfo, this, getHackDefaultProperties(reqX),
                propsToFetch -> createResource(authInfo, finalRealPath, getChildren,
                        reqX.getNonEmptyParameterO("offset").flatMapO(Cf.Integer::parseSafe),
                        reqX.getNonEmptyParameterO("amount").flatMapO(Cf.Integer::parseSafe),
                        reqX.getNonEmptyParameterO("sort"),
                        reqX.getNonEmptyParameterO("order").isSome("desc"),
                        propsToFetch)
        );
    }

    public ListF<DavPropertyName> getHackDefaultProperties(HttpServletRequestX reqX) {
        return WINDOWS_HACK_PROPERTY_NAMES
                .filter(x -> reqX.getUserAgent().filter(a -> a.startsWith("Microsoft-WebDAV-MiniRedir")).isPresent());
    }

    private MpfsPublicResource createPublicResource(AuthInfo authInfo, String hash) {
        return new MpfsPublicResource(
                authInfo,
                hash,
                mpfsClient.getPublicInfo(hash, Option.of(authInfo))
        );
    }

    private static String extractPath(DavResourceLocator locator) {
        String path = StringUtils.removeEnd(locator.getResourcePath(), "/");
        if (path == null || path.isEmpty()) {
            path = "/";
        }
        //TODO: may be we need it?
//        path = PathUtils.decodePath(path);
        while (path.startsWith("//")) {
            path = StringUtils.removeStart(path, "/");
        }
        return path;
    }

    @Override
    public DavResource createResource(DavResourceLocator locator, DavSession session) {
        throw new UnsupportedOperationException();
    }

    private <T> T withCallback(boolean isOurClient, MpfsCallback callback, Function<String, T> fn) {
        String callbackId = callbacksManager.registerCallback(isOurClient, callback);
        String callbackUrl = callbacksManager.getCallbackUrl(callbackId);
        try {
            return fn.apply(callbackUrl);
        } catch (RuntimeException e) {
            callbacksManager.findAndRemove(callbackId);
            throw e;
        }
    }

    public MpfsOperation asyncRm(AuthInfo uid, String realPath, Option<String> ifMatch,
            boolean isOurClient, MpfsCallback callback)
    {
        return withCallback(isOurClient, callback,
                (callbackUrl) -> mpfsClient.asyncRm(uid, realPath, ifMatch, callbackUrl)
        );
    }

    public MpfsOperation asyncTrashAppend(AuthInfo uid, String realPath, Option<String> ifMatch,
            boolean isOurClient, MpfsCallback callback)
    {
        return withCallback(isOurClient, callback,
                (callbackUrl) -> mpfsClient.asyncTrashAppend(uid, realPath, ifMatch, callbackUrl)
        );
    }

    public MpfsOperation asyncTrashClean(AuthInfo uid, boolean isOurClient, MpfsCallbackBase callback) {
        return withCallback(isOurClient, callback,
                (callbackUrl) -> mpfsClient.asyncTrashDropAll(uid, callbackUrl)
        );
    }

    public MpfsOperation asyncCopy(AuthInfo uid, String srcPath, String dstPath, boolean overwrite,
            boolean isOurClient, boolean doMkdirs, MpfsCallback callback)
    {
        return withMkdirsIfNeeded(uid, dstPath, () -> withCallback(isOurClient, callback,
                (callbackUrl) -> mpfsClient.asyncCopy(uid, srcPath, dstPath, overwrite, callbackUrl)), doMkdirs);
    }

    public MpfsOperation asyncMove(AuthInfo uid, String srcPath, String dstPath, boolean overwrite,
            boolean isOurClient, boolean doMkdirs, MpfsCallback callback)
    {
        return withMkdirsIfNeeded(uid, dstPath, () -> withCallback(isOurClient, callback,
                (callbackUrl) -> mpfsClient.asyncMove(uid, srcPath, dstPath, overwrite, callbackUrl)), doMkdirs);
    }

    public void mkDir(AuthInfo uid, String realPath) {
        mpfsClient.mkdir(uid, realPath);
    }

    public String mkSystemDir(AuthInfo uid, String type) {
        return mpfsClient.mksysdir(uid, type);
    }

    public MpfsStoreOperation store(Upload upload, AuthInfo user, String realPath, boolean isHidden,
            boolean isRedirectUpload, Option<String> userAgent, IpAddress remoteIp) throws DavException
    {
        Tuple2List<String, String> headers = getCustomHeaders(remoteIp);
        if (userAgent.isPresent()) {
            headers = headers.plus1(HttpHeaderNames.USER_AGENT, userAgent.get());
        }
        switch (upload.mode) {
            case RESUME:
                return mpfsClient.astore(
                        user,
                        realPath,
                        upload.md5.getOrThrow(() -> new DavException(400, "no md5 given")),
                        headers
                );
            case NORMAL:
                return mpfsClient.store(MpfsStoreOperationContext
                                .builder()
                                .uid(user)
                                .path(realPath)
                                .replaceMd5(upload.replaceMd5)
                                .md5(upload.md5)
                                .sha256(upload.sha256)
                                .livePhotoMd5(upload.livePhotoMd5)
                                .livePhotoSha256(upload.livePhotoSha256)
                                .livePhotoSize(upload.livePhotoSize)
                                .livePhotoType(upload.livePhotoType)
                                .livePhotoOperation(upload.livePhotoOperation)
                                .photostreamDestination(upload.photostreamDestination)
                                .deviceResourceSubtype(upload.deviceResourceSubtype)
                                .deviceCollections(upload.deviceCollections)
                                .deviceOriginalPath(upload.deviceOriginalPath)
                                .sourceId(upload.sourceId)
                                .forceDeletionLogDeduplication(upload.forceDeletionLogDeduplication)
                                .size(upload.size)
                                .ctime(upload.ctime)
                                .mtime(upload.mtime)
                                .etime(upload.etime)
                                .force(upload.force)
                                .isHidden(isHidden)
                                .isScreenshot(upload.isScreenshot)
                                .isPublic(upload.isPublic)
                                .iSwearIWillNeverPublishThisUrl(!isRedirectUpload)
                                .build(),
                        headers
                );
            case DELTA:
                return mpfsClient.dstore(
                        user,
                        realPath,
                        upload.replaceMd5.getOrThrow(() -> new DavException(400, "no md5 given")),
                        !isRedirectUpload
                );
            default:
                throw new IllegalStateException("Unknown upload mode: " + upload.mode);
        }
    }

    private void mkDirs(AuthInfo uid, String realPath) {
        String path = realPath;
        if (!realPath.endsWith("/")) {
            path = PathUtils.parentPath(path);
        }
        ListF<String> intermediates = Cf.arrayList();
        while (true) {
            String intermediate = path;
            if (intermediate.equals("/disk")) {
                break;
            }
            intermediates.add(intermediate);
            path = PathUtils.parentPath(path);
            if (path == null) {
                break;
            }
        }
        for (String intermediate : intermediates.reverse()) {
            try {
                mkDir(uid, intermediate);
            } catch (PermanentHttpFailureException e) {
                if (!e.getStatusCode().isSome(HttpStatus.SC_405_METHOD_NOT_ALLOWED)) {
                    throw e;
                }
            }
        }
    }

    private MpfsOperation withMkdirsIfNeeded(
            AuthInfo uid, String path, Function0<MpfsOperation> action, boolean doMkdirs)
    {
        try {
            return action.apply();
        } catch (PermanentHttpFailureException e) {
            if (doMkdirs && e.getStatusCode().isSome(HttpStatus.SC_409_CONFLICT)) {
                mkDirs(uid, path);
                return action.apply();
            } else {
                throw e;
            }
        }
    }

    public ListF<DavProperty> getProperties(Supplier<MpfsFileInfo> info, AuthInfo authInfo, DavPropertyName name) {
        if (!DavProperties.find(name).map(DavProperties.PropertyDescription::isAllowRead).getOrElse(true)) {
            // properties can't be read
            return null;
        }

        return propertiesFactories
                .filter(f -> f.accepts(name))
                .firstO()
                .map(f -> f.cons(info, authInfo, name))
                .filterNot(List::isEmpty)
                .getOrNull();
    }

    public MultiStatusResponse alterProperties(
            AuthInfo authInfo, MpfsResource mpfsResource, List<? extends PropEntry> changeList)
    {
        return propertiesSetter.alterProperties(authInfo, mpfsResource, changeList);
    }

    DavPropertySet getDefaultProperties(MpfsFileInfo info, AuthInfo authInfo) {
        DavPropertySet set = new DavPropertySet();

        DavProperties.all
                .filter(DavProperties.PropertyDescription::isDefault)
                .map(pd -> getProperties(() -> info, authInfo, pd.name))
                .filterNotNull()
                .flatMap(Function.identityF())
                .forEach(set::add);

        return set;
    }

    Option<MpfsOperation> getOperation(AuthInfo uid, String oid) {
        return mpfsClient.getOperation(uid, oid);
    }

    public String publish(AuthInfo uid, String path, IpAddress remoteIp) {
        return mpfsClient.getPublicUrl(mpfsClient.setPublic(uid, path, getCustomHeaders(remoteIp)));
    }

    public int unpublish(AuthInfo uid, String path) {
        return mpfsClient.setPrivate(uid, path).getStatusCode();
    }

    public int mobileSubscribe(AuthInfo uid, String token, Option<String> allow, String resources) {
        return mpfsClient.mobileSubscribe(uid, token, allow, resources);
    }

    public int mobileUnsubscribe(AuthInfo uid, String token) {
        return mpfsClient.mobileUnsubscribe(uid, token);
    }

    public Tuple2List<String, String> getCustomHeaders(IpAddress remoteIp) {
        return Tuple2List.fromPairs("X-Real-Ip", remoteIp.toString());
    }
}
