package ru.yandex.chemodan.app.webdav.repository;

import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.webdav.auth.AuthInfo;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.chemodan.mpfs.MpfsFileMetaDto;
import ru.yandex.chemodan.mpfs.MpfsResourceTimes;
import ru.yandex.chemodan.mpfs.MpfsShareFolderInvite;
import ru.yandex.misc.test.Assert;

public class ResourcesTest {

    @Test
    public void testMpfsInviteResource() {
        MpfsShareFolderInvite invite = new MpfsShareFolderInvite();
        invite.hash = "hash";
        invite.folderName = "folder";
        AuthInfo authInfo = AuthInfo.getDefault();
        MpfsInviteResource resource = new MpfsInviteResource(authInfo, invite);
        Assert.A.equals(invite, resource.getInvite());
        Assert.A.equals(authInfo, resource.getUser());
        Assert.A.equals("/?share/not_approved/hash", resource.getHref());
        Assert.A.equals("/?share/not_approved/hash", resource.getRealPath());
        Assert.A.equals("folder", resource.getDisplayName());
        Assert.A.equals(0, resource.getModificationTime());
        Assert.A.equals("OPTIONS, GET", resource.getSupportedMethods());
        Assert.A.equals(false, resource.isCollection());
        Assert.A.equals(
                "{DAV:}creationdate=1970-01-01T00:00:00Z, {DAV:}displayname=folder, {DAV:}getcontentlength=0, {DAV:}resourcetype=[], {urn:yandex:disk:meta}owner_name=null, {urn:yandex:disk:meta}readonly=false",
                StreamSupport
                        .stream(resource.getProperties().spliterator(), false)
                        .map(prop -> String.format("%s=%s", prop.getName(), prop.getValue()))
                        .sorted()
                        .collect(Collectors.joining(", "))
        );
        Assert.assertThrows(resource::getFileInfo, UnsupportedOperationException.class);
    }

    @Test
    public void testMpfsInviteListResource() {
        MpfsShareFolderInvite invite = new MpfsShareFolderInvite();
        invite.hash = "hash";
        invite.folderName = "folder";
        AuthInfo authInfo = AuthInfo.getDefault();
        MpfsInvitesListResource resource = new MpfsInvitesListResource(authInfo, Cf.list(invite));
        Assert.A.equals(authInfo, resource.getUser());
        Assert.A.equals("/?share/not_approved", resource.getHref());
        Assert.A.equals("/", resource.getDisplayName());
        Assert.A.equals(0, resource.getModificationTime());
        Assert.A.equals("OPTIONS, GET", resource.getSupportedMethods());
        Assert.A.equals(true, resource.isCollection());
        Assert.A.equals(
                "{DAV:}creationdate=2000-01-01T12:00:00Z, {DAV:}getlastmodified=Sat, 01 Jan 2000 12:00:00 GMT, {DAV:}resourcetype",
                StreamSupport
                        .stream(resource.getProperties().spliterator(), false)
                        .map(prop -> prop.getName().toString().equals("{DAV:}resourcetype") ?
                                prop.getName().toString() : String.format("%s=%s", prop.getName(), prop.getValue()))
                        .sorted()
                        .collect(Collectors.joining(", "))
        );
        Assert.A.equals(1, resource.getMembers().size());
        Assert.assertThrows(resource::getFileInfo, UnsupportedOperationException.class);
        Assert.assertThrows(resource::getRealPath, UnsupportedOperationException.class);
    }

    @Test
    public void testMpfsPublicResource() {
        AuthInfo authInfo = AuthInfo.getDefault();
        MpfsFileMetaDto meta = MpfsFileMetaDto.builder().build();
        MpfsFileInfo fileInfo = new MpfsFileInfo("name", "type", meta);
        MpfsPublicResource resource = new MpfsPublicResource(authInfo, "hash", fileInfo);
        Assert.A.equals(authInfo, resource.getUser());
        Assert.A.equals(fileInfo, resource.getFileInfo());
        Assert.A.equals("OPTIONS, GET, HEAD", resource.getSupportedMethods());
        Assert.A.equals(true, resource.isCollection());
        Assert.A.equals("name", resource.getDisplayName());
        Assert.A.equals("/?hash=hash", resource.getHref());
        Assert.A.equals(0, resource.getModificationTime());
        Assert.A.equals(authInfo, resource.getUser());
        Assert.A.equals(true, resource.exists());
        Assert.assertThrows(resource::getRealPath, UnsupportedOperationException.class);
        Assert.assertThrows(resource::getSession, UnsupportedOperationException.class);
        Assert.assertThrows(resource::getFactory, UnsupportedOperationException.class);
        Assert.assertThrows(resource::getLocks, UnsupportedOperationException.class);
        Assert.assertThrows(resource::getMembers, UnsupportedOperationException.class);
        Assert.assertThrows(resource::getCollection, UnsupportedOperationException.class);
        Assert.assertThrows(resource::getProperties, UnsupportedOperationException.class);
        Assert.assertThrows(resource::getPropertyNames, UnsupportedOperationException.class);
        Assert.assertThrows(resource::getResourcePath, UnsupportedOperationException.class);
        Assert.assertThrows(resource::getLocator, UnsupportedOperationException.class);
        Assert.assertThrows(() -> resource.addLockManager(null), UnsupportedOperationException.class);
        Assert.assertThrows(() -> resource.unlock(null), UnsupportedOperationException.class);
        Assert.assertThrows(() -> resource.refreshLock(null, null), UnsupportedOperationException.class);
        Assert.assertThrows(() -> resource.lock(null), UnsupportedOperationException.class);
        Assert.assertThrows(() -> resource.getLock(null, null), UnsupportedOperationException.class);
        Assert.assertThrows(() -> resource.hasLock(null, null), UnsupportedOperationException.class);
        Assert.assertThrows(() -> resource.isLockable(null, null), UnsupportedOperationException.class);
        Assert.assertThrows(() -> resource.copy(null, false), UnsupportedOperationException.class);
        Assert.assertThrows(() -> resource.move(null), UnsupportedOperationException.class);
        Assert.assertThrows(() -> resource.removeMember(null), UnsupportedOperationException.class);
        Assert.assertThrows(() -> resource.addMember(null, null), UnsupportedOperationException.class);
        Assert.assertThrows(() -> resource.alterProperties(null), UnsupportedOperationException.class);
        Assert.assertThrows(() -> resource.removeProperty(null), UnsupportedOperationException.class);
        Assert.assertThrows(() -> resource.setProperty(null), UnsupportedOperationException.class);
        Assert.assertThrows(() -> resource.getProperties(null), UnsupportedOperationException.class);
        Assert.assertThrows(() -> resource.getProperty(null), UnsupportedOperationException.class);
        Assert.assertThrows(() -> resource.spool(null), UnsupportedOperationException.class);
    }

    @Test
    public void testUnloadedResource() {
        AuthInfo authInfo = AuthInfo.getDefault();
        MpfsFileMetaDto meta = MpfsFileMetaDto.builder().build();
        MpfsFileInfo fileInfo =
                new MpfsFileInfo(Option.of("name"), Option.of("type"), Option.of("path"), meta, MpfsResourceTimes.ZERO);
        MpfsRealResourceBase base = new FileResource(authInfo, fileInfo, Cf.list(), null);
        Assert.A.equals("name", base.getDisplayName());
        Assert.A.equals(0, base.getModificationTime());
        Assert.A.equals(false, base.getMembers().hasNext());
        MpfsUnloadedResource resource = new MpfsUnloadedResource("path", authInfo, null, Cf.list(), (props) -> base);
        Assert.assertThrows(() -> resource.addLockManager(null), UnsupportedOperationException.class);
        Assert.assertThrows(() -> resource.unlock(null), UnsupportedOperationException.class);
        Assert.assertThrows(() -> resource.refreshLock(null, null), UnsupportedOperationException.class);
        Assert.assertThrows(() -> resource.lock(null), UnsupportedOperationException.class);
        Assert.assertThrows(() -> resource.getLock(null, null), UnsupportedOperationException.class);
        Assert.assertThrows(() -> resource.hasLock(null, null), UnsupportedOperationException.class);
        Assert.assertThrows(() -> resource.isLockable(null, null), UnsupportedOperationException.class);
        Assert.assertThrows(() -> resource.copy(null, false), UnsupportedOperationException.class);
        Assert.assertThrows(() -> resource.move(null), UnsupportedOperationException.class);
        Assert.assertThrows(() -> resource.removeMember(null), UnsupportedOperationException.class);
        Assert.assertThrows(() -> resource.addMember(null, null), UnsupportedOperationException.class);
        Assert.assertThrows(() -> resource.alterProperties(null), UnsupportedOperationException.class);
        Assert.assertThrows(() -> resource.removeProperty(null), UnsupportedOperationException.class);
        Assert.assertThrows(() -> resource.setProperty(null), UnsupportedOperationException.class);
        Assert.assertThrows(() -> resource.spool(null), UnsupportedOperationException.class);
    }

}
