package ru.yandex.chemodan.app.webdav.repository.properties;

import org.apache.jackrabbit.webdav.property.DavProperty;
import org.apache.jackrabbit.webdav.property.DavPropertyName;
import org.apache.jackrabbit.webdav.property.DefaultDavProperty;
import org.apache.jackrabbit.webdav.xml.Namespace;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.reflection.ClassX;

import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.PropertyDescription.otherProperty;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.PropertyDescription.propertyFromMeta;

/**
 * @author tolmalev
 */
public class DavProperties {
    public static Namespace META = Namespace.getNamespace("urn:yandex:disk:meta");
    public static Namespace USER_STATES = Namespace.getNamespace("urn:yandex:user:state");
    public static Namespace SHARE_USERS = Namespace.getNamespace("urn:yandex:disk:share:users");
    public static Namespace QUIRKS = Namespace.getNamespace("urn:yandex:disk:quirks");
    public static Namespace OPERATIONS = Namespace.getNamespace("urn:yandex:disk:operations");

    public static Namespace MS = Namespace.getNamespace("urn:schemas-microsoft-com:");

    public static Namespace DAV = Namespace.getNamespace("d", "DAV:");
    public static Namespace EMPTY_NS = Namespace.EMPTY_NAMESPACE;


    public static PropertyDescription SIZE = propertyFromMeta(EMPTY_NS, "size", "size", Tag.READ);
    public static PropertyDescription FILE_URL = propertyFromMeta(EMPTY_NS, "file_url", "file_url", Tag.DEFAULT, Tag.READ);
    public static PropertyDescription DIGEST_URL = propertyFromMeta(EMPTY_NS, "digest_url", "digest_url", Tag.READ);
    public static PropertyDescription MULCA_FILE_URL =
            propertyFromMeta(EMPTY_NS, "mulca_file_url", "file_mid", Tag.DEFAULT, Tag.READ);
    public static PropertyDescription MULCA_DIGEST_URL =
            propertyFromMeta(EMPTY_NS, "mulca_digest_url", "digest_mid",Tag.DEFAULT, Tag.READ);
    public static PropertyDescription CREATION_TIME =
            otherProperty(EMPTY_NS, "creation_time", Tag.READ, Tag.WRITE, Tag.NOREMOVE);

    // http://webdav.org/specs/rfc4918.html#dav.properties
    public static PropertyDescription GETETAG = propertyFromMeta(DAV, "getetag", "md5", Tag.READ, Tag.DEFAULT);
    public static PropertyDescription GETCONTENTTYPE =
            propertyFromMeta(DAV, "getcontenttype", "mimetype", Tag.READ, Tag.DEFAULT);
    public static PropertyDescription RESOURCETYPE = otherProperty(DAV, "resourcetype", Tag.READ, Tag.DEFAULT);
    public static PropertyDescription CREATIONDATE =
            otherProperty(DAV, "creationdate", Tag.READ, Tag.WRITE, Tag.NOREMOVE, Tag.DEFAULT);
    public static PropertyDescription GETCONTENTLENGTH =
            propertyFromMeta(DAV, "getcontentlength", "size", Tag.READ, Tag.DEFAULT);
    public static PropertyDescription GETLASTMODIFIED = otherProperty(DAV, "getlastmodified", Tag.READ, Tag.DEFAULT);
    public static PropertyDescription DISPLAYNAME = otherProperty(DAV, "displayname", Tag.READ, Tag.DEFAULT);

    // http://webdav.org/specs/rfc4437.html#properties
    public static PropertyDescription REFTARGET =
            propertyFromMeta(DAV, "reftarget", "fs_symbolic_link", Tag.READ, Tag.DEFAULT);

    // DAV quota properties: http://tools.ietf.org/html/rfc4331
    public static PropertyDescription QUOTA_AVAILABLE_BYTES = otherProperty(DAV, "quota-available-bytes", Tag.READ);
    public static PropertyDescription QUOTA_USED_BYTES = otherProperty(DAV, "quota-used-bytes", Tag.READ);

    // These are non-standard properties, used by Microsoft, see http://www.ics.uci.edu/~ejw/authoring/props/draft-hopmann-collection-props-00.txt
    public static PropertyDescription ISHIDDEN = propertyFromMeta(DAV, "ishidden", "visible", Tag.READ, Tag.WRITE);
    public static PropertyDescription HASSUBS = propertyFromMeta(DAV, "hassubs", "hasfolders", Tag.READ);
    public static PropertyDescription CHILDCOUNT = propertyFromMeta(DAV, "childcount", "numfolders", Tag.READ);
    public static PropertyDescription OBJECTCOUNT =
            propertyFromMeta(DAV, "objectcount", Cf.list("numfiles", "numfolders"), Tag.READ);

    public static PropertyDescription MS_ATTR =
            propertyFromMeta(MS, "Win32FileAttributes",
                    DavPropertyNameUtils.getCustomPropertyNameSafe(DavPropertyName.create("Win32FileAttributes", MS)),
                    Tag.READ, Tag.WRITE);

    // Publish API: http://api.yandex.ru/disk/doc/dg/reference/publish.xml
    public static PropertyDescription PUBLIC_URL =
            propertyFromMeta(META, "public_url", "short_url", Tag.READ, Tag.WRITE);

    // Extended MPFS space properties
    public static PropertyDescription QUOTA_LIMIT_BYTES = otherProperty(META, "quota-limit-bytes", Tag.READ);
    public static PropertyDescription MAX_FILE_SIZE = otherProperty(META, "max-file-size", Tag.READ);
    public static PropertyDescription PAID_MAX_FILE_SIZE = otherProperty(META, "paid-max-file-size", Tag.READ);
    public static PropertyDescription TRASH_USED_BYTES = otherProperty(META, "trash-used-bytes", Tag.READ);

    // Default folders (awkward localization): http://wiki.yandex-team.ru/disk/webdav#putikdefoltnympapkam
    public static PropertyDescription DEFAULT_FOLDERS = otherProperty(META, "default-folders", Tag.READ);
    // Special photostream property for accessing client-specific settings: http://wiki.yandex-team.ru/disk/webdav#avtozagruzkefotografijj
    public static PropertyDescription PHOTOSTREAM = otherProperty(META, "photostream", Tag.READ, Tag.WRITE);

    // Extended resource attributes
    public static PropertyDescription VISIBLE = propertyFromMeta(META, "visible", "visible", Tag.READ, Tag.WRITE);
    public static PropertyDescription EMPTY = propertyFromMeta(META, "empty", "empty", Tag.READ);
    public static PropertyDescription SHA256 = propertyFromMeta(META, "sha256", "sha256", Tag.READ);
    public static PropertyDescription ETIME = otherProperty(META, "etime", Tag.READ); // CHEMODAN-11816
    public static PropertyDescription MEDIATYPE = propertyFromMeta(META, "mediatype", "mediatype", Tag.READ);
    // CHEMODAN-14521
    public static PropertyDescription PHOTOSLICE_ALBUM_TYPE =
            propertyFromMeta(META, "photoslice_album_type", "photoslice_album_type", Tag.READ); //CHEMODAN-64011
    public static PropertyDescription ALBUM_EXCLUSIONS =
            propertyFromMeta(META, "albums_exclusions", "albums_exclusions", Tag.READ);
    //CHEMODAN-65541
    public static PropertyDescription HASTHUMBNAIL = propertyFromMeta(META, "hasthumbnail", "pmid", Tag.READ);
    // CHEMODAN-15557
    public static PropertyDescription FOLDER_TYPE = propertyFromMeta(META, "folder_type", "folder_type", Tag.READ);
    public static PropertyDescription PHOTOSLICE_TIME =
            propertyFromMeta(META, "photoslice_time", "photoslice_time", Tag.READ);
    public static PropertyDescription VIDEO_DURATION_MILLIS =
            propertyFromMeta(META, "video_duration_millis", "video_info", Tag.READ);
    public static PropertyDescription BEAUTY = propertyFromMeta(META, "beauty", "aesthetics", Tag.READ);
    public static PropertyDescription WIDTH = propertyFromMeta(META, "width", "width", Tag.READ);
    public static PropertyDescription HEIGHT = propertyFromMeta(META, "height", "height", Tag.READ);

    // Sharing: https://wiki.yandex-team.ru/disk/webdav#obshhiepapki
    public static PropertyDescription SHARED = propertyFromMeta(META, "shared", "group", Tag.READ);
    public static PropertyDescription OWNED = propertyFromMeta(META, "owned", "group", Tag.READ);
    public static PropertyDescription READONLY = propertyFromMeta(META, "readonly", "group", Tag.READ);
    public static PropertyDescription OWNER_NAME = propertyFromMeta(META, "owner_name", "group", Tag.READ);

    public static PropertyDescription MPFS_FILE_ID = propertyFromMeta(META, "mpfs_file_id", "file_id", Tag.READ);
    public static PropertyDescription MPFS_RESOURCE_ID =
            propertyFromMeta(META, "mpfs_resource_id", "resource_id", Tag.READ);
    public static PropertyDescription FLAG_BROKEN = otherProperty(META, "flag_broken", Tag.WRITE);

    // Quirks
    // CHEMODAN-17013: request for rehashing
    public static PropertyDescription QUIRKS_REHASH_SIZE = otherProperty(QUIRKS, "rehash-size", Tag.WRITE);
    public static PropertyDescription QUIRKS_REHASH_MD5 = otherProperty(QUIRKS, "rehash-md5", Tag.WRITE);
    public static PropertyDescription QUIRKS_REHASH_SHA256 = otherProperty(QUIRKS, "rehash-sha256", Tag.WRITE);

    public static PropertyDescription ALIAS_ENABLED =
            propertyFromMeta(META, "alias_enabled", "alias_enabled", Tag.READ);

    public static PropertyDescription STATE_ALLPROP = otherProperty(USER_STATES, "allprop", Tag.READ);

    public static ListF<PropertyDescription> all = ClassX.wrap(DavProperties.class)
            .getStaticFieldsOfType(PropertyDescription.class)
            .map(fieldX -> fieldX.get(null))
            .cast();

    public static MapF<DavPropertyName, PropertyDescription> allMap = all.toMapMappingToKey(pd -> pd.name);

    public static Option<PropertyDescription> find(DavPropertyName name) {
        return allMap.getO(name);
    }


    public static class PropertyDescription extends DefaultObject {
        public final DavPropertyName name;
        public final Option<ListF<String>> mpfsFileMetaNames;
        private final SetF<Tag> tags;

        public static PropertyDescription propertyFromMeta(
                Namespace namespace, String name, String mpfsFileMetaName, Tag... tags)
        {
            return new PropertyDescription(DavPropertyName.create(name, namespace),
                    Option.of(Cf.list(mpfsFileMetaName)), tags);
        }

        public static PropertyDescription propertyFromMeta(
                Namespace namespace, String name, ListF<String> mpfsFileMetaNames, Tag... tags)
        {
            return new PropertyDescription(DavPropertyName.create(name, namespace),
                    Option.of(mpfsFileMetaNames), tags);
        }

        public static PropertyDescription otherProperty(Namespace namespace, String name, Tag... tags) {
            return new PropertyDescription(DavPropertyName.create(name, namespace),
                    Option.empty(), tags);
        }

        private PropertyDescription(DavPropertyName name, Option<ListF<String>> mpfsFileMetaNames, Tag... tags) {
            this.name = name;
            this.tags = Cf.set(tags);
            this.mpfsFileMetaNames = mpfsFileMetaNames;
        }

        public boolean isAllowRead() {
            return tags.containsTs(Tag.READ);
        }

        public boolean isAllowWrite() {
            return tags.containsTs(Tag.WRITE);
        }

        public boolean isAllowRemove() {
            return tags.containsTs(Tag.WRITE) && !tags.containsTs(Tag.NOREMOVE);
        }

        public boolean isDefault() {
            return tags.containsTs(Tag.DEFAULT);
        }

        public DavPropertyName getName() {
            return name;
        }

        public Option<ListF<String>> getMpfsFileMetaNames() {
            return mpfsFileMetaNames;
        }

        public <T> DavProperty<T> cons(T value) {
            return new DefaultDavProperty<T>(name, value);
        }
    }

    private enum Tag {
        READ,
        WRITE,
        NOREMOVE,
        DEFAULT;
    }
}
