package ru.yandex.chemodan.app.webdav.repository.properties;

import java.util.Date;
import java.util.function.Supplier;

import com.fasterxml.jackson.databind.util.ISO8601Utils;
import org.apache.http.client.utils.DateUtils;
import org.apache.jackrabbit.webdav.property.DavProperty;
import org.apache.jackrabbit.webdav.property.DavPropertyName;
import org.apache.jackrabbit.webdav.property.ResourceType;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.webdav.auth.AuthInfo;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.chemodan.mpfs.MpfsFileMeta;
import ru.yandex.chemodan.mpfs.MpfsResourceTimes;
import ru.yandex.commune.video.format.FileInformation;
import ru.yandex.misc.dataSize.DataSize;

import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.ALBUM_EXCLUSIONS;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.ALIAS_ENABLED;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.BEAUTY;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.CHILDCOUNT;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.CREATIONDATE;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.CREATION_TIME;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.DIGEST_URL;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.DISPLAYNAME;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.EMPTY;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.ETIME;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.FILE_URL;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.FOLDER_TYPE;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.GETCONTENTLENGTH;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.GETCONTENTTYPE;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.GETETAG;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.GETLASTMODIFIED;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.HASSUBS;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.HASTHUMBNAIL;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.HEIGHT;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.ISHIDDEN;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.MEDIATYPE;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.MPFS_FILE_ID;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.MPFS_RESOURCE_ID;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.MULCA_DIGEST_URL;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.MULCA_FILE_URL;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.OBJECTCOUNT;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.OWNED;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.OWNER_NAME;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.PHOTOSLICE_ALBUM_TYPE;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.PHOTOSLICE_TIME;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.PUBLIC_URL;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.READONLY;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.REFTARGET;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.RESOURCETYPE;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.SHA256;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.SHARED;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.SIZE;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.VIDEO_DURATION_MILLIS;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.VISIBLE;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.WIDTH;

/**
 * @author tolmalev
 */
public class MetaPropertiesFactory implements PropertiesFactory {

    private static final Function<Integer, Integer> INT_BOOL = (i) -> i == 1 ? 1 : 0;
    private static final Function<Integer, Integer> INT_BOOL_INV = (i) -> i != 1 ? 1 : 0;

    private static final Function<MpfsFileInfo, MpfsFileMeta> META = MpfsFileInfo::getMeta;
    private static final Function<MpfsFileInfo, MpfsResourceTimes> TIMES = MpfsFileInfo::getTimes;

    private static <T> Function<T, Option<T>> toOption() {
        return Option::of;
    }

    private static <T> Function<Option<T>, Option<T>> orElse(T value) {
        return option -> option.orElse(Option.of(value));
    }

    private static <X, Y> Function<Option<X>, Option<Y>> map(Function<X, Y> function) {
        return option -> option.map(function);
    }

    public static final MapF<DavPropertyName, OnePropertyFactory<MpfsFileInfo>> properties = Cf.list(

            new OnePropertyFactory<>(SIZE, META.andThen(MpfsFileMeta::getSize).andThen(map(DataSize::toBytes))),
            new OnePropertyFactory<>(FILE_URL, META.andThen(MpfsFileMeta::getFileUrl)),
            new OnePropertyFactory<>(DIGEST_URL, META.andThen(MpfsFileMeta::getDigestUrl)),
            new OnePropertyFactory<>(MULCA_FILE_URL, META.andThen(MpfsFileMeta::getFileMid)),
            new OnePropertyFactory<>(MULCA_DIGEST_URL, META.andThen(MpfsFileMeta::getDigestMid)),
            new OnePropertyFactory<>(CREATION_TIME, TIMES.andThen(MpfsResourceTimes::getCtime).andThen(toOption())),

            new OnePropertyFactory<>(GETETAG, META.andThen(MpfsFileMeta::getMd5)),
            new OnePropertyFactory<>(GETCONTENTTYPE, META.andThen(MpfsFileMeta::getMimeType)),
            new OnePropertyFactory<MpfsFileInfo>(RESOURCETYPE, (info) ->
                    info.type.map(type -> "file".equals(type)
                            ? new ResourceType(ResourceType.DEFAULT_RESOURCE)
                            : new ResourceType(ResourceType.COLLECTION))
            ),
            new OnePropertyFactory<MpfsFileInfo>(CREATIONDATE, (info) -> Option.of(
                    ISO8601Utils.format(new Date(info.times.ctime * 1000)))),
            new OnePropertyFactory<MpfsFileInfo>(GETLASTMODIFIED, (info) -> Option.of(
                    DateUtils.formatDate(new Date(info.times.mtime * 1000)))),

            new OnePropertyFactory<>(GETCONTENTLENGTH, META.andThen(MpfsFileMeta::getSize)),
            new OnePropertyFactory<>(DISPLAYNAME, MpfsFileInfo::getName),

            // http://webdav.org/specs/rfc4437.html#properties
            new OnePropertyFactory<>(REFTARGET, META.andThen(MpfsFileMeta::getFs_symbolic_link)),

            new OnePropertyFactory<>(ISHIDDEN, META.andThen(MpfsFileMeta::getVisible).andThen(map(INT_BOOL_INV))),
            new OnePropertyFactory<>(HASSUBS, META.andThen(MpfsFileMeta::getHasfolders).andThen(map(INT_BOOL))),
            new OnePropertyFactory<>(CHILDCOUNT, META.andThen(MpfsFileMeta::getNumfolders)),
            new OnePropertyFactory<>(OBJECTCOUNT, META.andThen((Function<MpfsFileMeta, Integer>) meta ->
                    meta.getNumfiles().getOrElse(0) + meta.getNumfolders().getOrElse(0)).andThen(toOption())),

            new OnePropertyFactory<>(PUBLIC_URL, META.andThen(MpfsFileMeta::getShortUrl)),

            new OnePropertyFactory<>(VISIBLE, META.andThen(MpfsFileMeta::getVisible).andThen(map(INT_BOOL))),

            new OnePropertyFactory<>(EMPTY, META.andThen(MpfsFileMeta::getEmpty)),

            new OnePropertyFactory<>(SHA256, META.andThen(MpfsFileMeta::getSha256)),
            new OnePropertyFactory<>(ETIME, TIMES.andThen(MpfsResourceTimes::getEtime)),
            new OnePropertyFactory<>(MEDIATYPE, META.andThen(MpfsFileMeta::getMediaType)),
            new OnePropertyFactory<>(PHOTOSLICE_ALBUM_TYPE, META.andThen(MpfsFileMeta::getPhotoslice_album_type)),
            new OnePropertyFactory<>(ALBUM_EXCLUSIONS, META.andThen((Function<MpfsFileMeta, Option<String>>) meta ->
                    Option.when(meta.getAlbums_exclusions().isNotEmpty(), meta.getAlbums_exclusions().mkString(",")))
            ),
            new OnePropertyFactory<>(HASTHUMBNAIL, META.andThen(MpfsFileMeta::getPmid)
                    .andThen((Function<Option<String>, Integer>) id -> id.isPresent() ? 1 : 0)
                    .andThen(toOption())),
            new OnePropertyFactory<>(FOLDER_TYPE, META.andThen(MpfsFileMeta::getFolderType)),
            new OnePropertyFactory<>(PHOTOSLICE_TIME, META.andThen(MpfsFileMeta::getPhotosliceTime)),
            new OnePropertyFactory<>(BEAUTY, META.andThen(MpfsFileMeta::getAesthetics)),
            new OnePropertyFactory<>(WIDTH, META.andThen(MpfsFileMeta::getWidth)),
            new OnePropertyFactory<>(HEIGHT, META.andThen(MpfsFileMeta::getHeight)),
            new OnePropertyFactory<>(VIDEO_DURATION_MILLIS, META.andThen(new Function<MpfsFileMeta, Option<Long>>()

            {
                @Override
                public Option<Long> apply(MpfsFileMeta mpfsFileMeta) {
                    return mpfsFileMeta.getVideoInfo().flatMapO(FileInformation::getDuration).map(Duration::getMillis);
                }
            })),

            new OnePropertyFactory<>(MPFS_FILE_ID, META.andThen(MpfsFileMeta::getFileId)),
            new OnePropertyFactory<>(MPFS_RESOURCE_ID,
                    META.andThen(MpfsFileMeta::getResourceId).andThen(map(Object::toString))),

            new OnePropertyFactory<>(ALIAS_ENABLED, META.andThen(MpfsFileMeta::getAlias_enabled)),

            //{ {?META, "shared", str_bool}, {meta, {"group", ["is_shared"]}, int_bool}, [read, {fallback, false}] },
            new OnePropertyFactory<>(SHARED, META
                    .andThen(MpfsFileMeta::getGroup)
                    .andThen(map(group -> group.is_shared == 1))
                    .andThen(orElse(false))
                    .andThen(map(Object::toString))
            ),
            //{ {?META, "owned", str_bool}, {meta, {"group", ["is_owned"]}, int_bool}, [read, {fallback, true}] },
            new OnePropertyFactory<>(OWNED, META
                    .andThen(MpfsFileMeta::getGroup)
                    .andThen(map(group -> group.is_owned == 1))
                    .andThen(orElse(true))
                    .andThen(map(Object::toString))
            ),

            //{ {?META, "readonly", str_bool}, {meta, [{"group", ["is_owned"]}, {"group", ["rights"]}], share_readonly}, [read] },
            new OnePropertyFactory<>(READONLY, META
                    .andThen(MpfsFileMeta::getGroup)
                    .andThen(map(group -> group.is_owned != 1 && !group.rights.isSome(660)))
                    .andThen(orElse(false))
                    .andThen(map(Object::toString))
            ),
            //{ {?META, "owner_name", entity_escaped}, {meta, {"group", ["owner", "username"]}, identity}, [read] },
            new OnePropertyFactory<>(OWNER_NAME,
                    META.andThen(MpfsFileMeta::getGroup).andThen(map(group -> group.owner.username.getOrElse(""))))).
            toMapMappingToKey(OnePropertyFactory::getName);

    @Override
    public int order() {
        return 0;
    }

    @Override
    public boolean accepts(DavPropertyName name) {
        return properties.containsKeyTs(name);
    }

    @Override
    public Option<DavProperty> cons(Supplier<MpfsFileInfo> info, AuthInfo authInfo, DavPropertyName name) {
        return properties.getO(name).flatMapO(factory -> factory.cons(info.get(), authInfo));
    }

    @Override
    public ListF<String> getMetaFields(DavPropertyName name) {
        return DavProperties.find(name).flatMapO(DavProperties.PropertyDescription::getMpfsFileMetaNames).getOrElse(Cf::list);
    }
}
