package ru.yandex.chemodan.app.webdav.repository.properties;

import java.io.ByteArrayOutputStream;
import java.text.ParseException;
import java.text.ParsePosition;
import java.util.ArrayList;
import java.util.List;

import com.fasterxml.jackson.databind.util.ISO8601Utils;
import lombok.AllArgsConstructor;
import org.apache.jackrabbit.webdav.MultiStatusResponse;
import org.apache.jackrabbit.webdav.property.DavProperty;
import org.apache.jackrabbit.webdav.property.DavPropertyName;
import org.apache.jackrabbit.webdav.property.DefaultDavProperty;
import org.apache.jackrabbit.webdav.property.PropEntry;
import org.apache.jackrabbit.webdav.xml.Namespace;
import org.w3c.dom.Element;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.webdav.auth.AuthInfo;
import ru.yandex.chemodan.app.webdav.repository.MpfsResource;
import ru.yandex.chemodan.mpfs.MpfsCallbackResponse;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.util.exception.PermanentHttpFailureException;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.xml.dom.DomUtils;

/**
 * @author tolmalev
 */
public class PropertiesSetter {

    private final MpfsClient mpfsClient;

    private final MapF<DavPropertyName, PropertyGroup> groupsByName = Cf.map(
            DavProperties.PUBLIC_URL, PropertyGroup.PUBLIC_URL,
            DavProperties.PHOTOSTREAM, PropertyGroup.SETTING
    ).entries().map1(pd -> pd.name).toMap();

    private final MapF<Namespace, PropertyGroup> groupsByNamespace = Tuple2List.<Namespace, PropertyGroup>fromPairs(
            DavProperties.META, PropertyGroup.RESOURCE,
            DavProperties.SHARE_USERS, PropertyGroup.SHARE_USER,
            DavProperties.USER_STATES, PropertyGroup.USER_STATE,
            DavProperties.QUIRKS, PropertyGroup.QUIRK_REHASH,
            DavProperties.DAV, PropertyGroup.RESOURCE,
            DavProperties.MS, PropertyGroup.MS
    ).toMap();

    private final MapF<PropertyGroup, GroupApplier> appliers = Tuple2List.<PropertyGroup, GroupApplier>fromPairs(
            PropertyGroup.PUBLIC_URL, new SingleApplierWithNoPreCheckRecodeAuth() {
                @Override
                protected MpfsCallbackResponse set(MpfsResource resource, Operation operation) {
                    return mpfsClient.setPublic(resource.getUser(), resource.getRealPath());
                }

                @Override
                protected MpfsCallbackResponse remove(MpfsResource resource, Operation operation) {
                    return mpfsClient.setPrivate(resource.getUser(), resource.getRealPath());
                }

                @Override
                protected PatchResult setPostProcess(Function0<MpfsCallbackResponse> setF) {
                    MpfsCallbackResponse mpfsResponse = setF.apply();
                    return new PatchResult(mpfsResponse.getStatusCode(), mpfsClient.getPublicUrl(mpfsResponse));
                }
            },
            PropertyGroup.USER_STATE, new SingleApplierWithNoPreCheckRecodeAuth() {
                @Override
                protected MpfsCallbackResponse set(MpfsResource resource, Operation operation) {
                    return mpfsClient.stateSet(resource.getUser(), operation.getName(), operation.getStringValue());
                }

                @Override
                protected MpfsCallbackResponse remove(MpfsResource resource, Operation operation) {
                    return mpfsClient.stateRemove(resource.getUser(), operation.getName());
                }
            },
            PropertyGroup.SHARE_USER, new SingleApplierWithNoPreCheckRecodeAuth() {
                @Override
                protected MpfsCallbackResponse set(MpfsResource resource, Operation operation) {
                    String login = operation.getName();
                    String rightsStr = operation.getStringValue();

                    int rightsInt;

                    switch (rightsStr) {
                        case "rw": rightsInt = 660; break;
                        case "r": rightsInt = 640; break;
                        default: return consStatusResponse(HttpStatus.SC_409_CONFLICT);
                    }

                    return mpfsClient.shareInviteUser(resource.getUser(), resource.getRealPath(), rightsInt, login);
                }

                @Override
                protected MpfsCallbackResponse remove(MpfsResource resource, Operation operation) {
                    //unsupported
                    return consStatusResponse(HttpStatus.SC_400_BAD_REQUEST);
                }
            },
            PropertyGroup.RESOURCE, new SetPropGroupApplier() {
                @Override
                protected PatchSemiResult propertyRecode(
                        AuthInfo authInfo, DavPropertyName name, Option<DavProperty> property)
                {
                    String key;
                    Option<String> value;

                    if (name.equals(DavProperties.CREATIONDATE.name)) {
                        try {
                            key = "ctime";
                            value = Option.when(property.isPresent(),
                                    String.valueOf(ISO8601Utils.parse(
                                            (String) property.get().getValue(), new ParsePosition(0))
                                            .getTime() / 1000));
                        } catch (ParseException e) {
                            return PatchSemiResult.error(name, HttpStatus.SC_409_CONFLICT);
                        }
                    } else if (name.equals(DavProperties.VISIBLE.name)) {
                        key = "visible";
                        value = property.map(p -> (String) p.getValue());
                    } else if (name.equals(DavProperties.FLAG_BROKEN.name)) {
                        key = "broken";
                        value = property.map(p -> (String) p.getValue());
                    } else {
                        return super.propertyRecode(authInfo, name, property);
                    }

                    return new PatchSemiResult(Option.of(name), Option.of(key), value, Option.empty(), Option.empty());
                }
            },
            PropertyGroup.MS, new GroupApplier() {
                @Override
                public void apply(AuthInfo authInfo, MpfsResource resource, ListF<Operation> operations,
                        MultiStatusResponse response)
                {
                    operations.forEach(operation -> {
                        DavPropertyName name = operation.name;
                        Option<DavProperty> property = operation.value;

                        if (name.equals(DavProperties.MS_ATTR.name)) {
                            PatchSemiResult win32Attr = super.propertyRecode(authInfo, name, property);

                            if (operation.isSet()) {
                                Option<String> value = property.map(p ->
                                        (Integer.parseInt((String) p.getValue(), 16) & 0x02) == 0 ? "1" : "0");
                                PatchSemiResult visibility = new PatchSemiResult(
                                        Option.of(name), Option.of("visible"), value, Option.empty(), Option.empty());

                                ListF<PatchSemiResult> set = Cf.list(visibility, win32Attr);

                                int status = mpfsClient.setProp(
                                        resource.getUser(), resource.getRealPath(),
                                        set.map(PatchSemiResult::getSetKeyValue)
                                                .toTuple2List(Tuple2::get1, Tuple2::get2),
                                        Cf.list()).getStatusCode();

                                addToResponse(name, operation.getStringValue(), status, response);
                            } else if (operation.isRemove()) {
                                int status = mpfsClient.setProp(
                                        resource.getUser(), resource.getRealPath(),
                                        Tuple2List.fromPairs(),
                                        Cf.list(win32Attr.getRemoveKey())).getStatusCode();
                                addToResponse(name, "", status, response);
                            }
                        }
                    });
                }
            },
            PropertyGroup.OTHER, new SetPropGroupApplier() {
                @Override
                protected PatchSemiResult operationProcess(AuthInfo authInfo, Operation operation) {
                    PatchSemiResult semiResult = propertyRecode(authInfo, operation.name, operation.value);

                    if (semiResult.isError()
                            || (semiResult.isSet() && operation.isSet())
                            || (semiResult.isRemove() && operation.isRemove()))
                    {
                        return semiResult;
                    } else {
                        return PatchSemiResult.error(operation.name, HttpStatus.SC_409_CONFLICT);
                    }
                }
            },
            PropertyGroup.SETTING, new SingleApplierWithNamespaceKeyValue() {

                @Override
                protected MpfsCallbackResponse set(
                        MpfsResource resource, Operation operation, String namespace, String key, String value)
                {
                    return mpfsClient.settingSet(resource.getUser(), namespace, key, value);
                }

                @Override
                protected MpfsCallbackResponse remove(
                        MpfsResource resource, Operation operation, String namespace, String key)
                {
                    return mpfsClient.settingRemove(resource.getUser(), namespace, key);
                }

                @Override
                protected PatchSemiResult propertyRecode(
                        AuthInfo authInfo, DavPropertyName name, Option<DavProperty> property)
                {
                    String key;
                    Option<String> value;
                    String namespace = name.getName();

                    if (name.equals(DavProperties.PHOTOSTREAM.name)) {
                        key = authInfo.ourClient.filterMap(ourClient -> ourClient.installId).getOrElse("");
                        value = property.map(PropertiesSetter::serializeStringOrXmlDavProperty);
                    } else {
                        return super.propertyRecode(authInfo, name, property);
                    }

                    return new PatchSemiResult(Option.of(name), Option.of(key), value, Option.empty(), Option.empty())
                            .withNamespace(namespace);
                }
            }

    ).toMap();

    public PropertiesSetter(MpfsClient mpfsClient) {
        this.mpfsClient = mpfsClient;
    }

    private PropertyGroup getGroup(DavPropertyName name) {
        return groupsByName
                .getO(name)
                .orElse(groupsByNamespace.getO(name.getNamespace()))
                .getOrElse(PropertyGroup.OTHER);
    }

    private GroupApplier getApplier(PropertyGroup group) {
        return appliers.getTs(group);
    }

    private void apply(AuthInfo authInfo, PropertyGroup group, MpfsResource mpfsResource,
            ListF<Operation> ops, MultiStatusResponse response)
    {
        getApplier(group).apply(authInfo, mpfsResource, ops, response);
    }

    public MultiStatusResponse alterProperties(
            AuthInfo authInfo, MpfsResource mpfsResource, List<? extends PropEntry> changeList)
    {
        ListF<Operation> operations = Cf.x(changeList).map(PropertiesSetter::parse);

        MultiStatusResponse response = new MultiStatusResponse(mpfsResource.getHref(), null);

        operations
                .groupBy(op -> getGroup(op.name))
                .entries().sortedBy1()
                .forAll((group, ops) -> {
                    apply(authInfo, group, mpfsResource, ops, response);
                    return true;
                });

        return response;
    }

    private static Operation parse(PropEntry propEntry) {
        if (propEntry instanceof DavPropertyName) {
            return new Operation(
                    (DavPropertyName) propEntry,
                    OperationType.REMOVE,
                    Option.empty()
            );
        } else if (propEntry instanceof DavProperty) {
            return new Operation(
                    ((DavProperty) propEntry).getName(),
                    OperationType.SET,
                    Option.of((DavProperty) propEntry)
            );
        } else {
            throw new UnsupportedOperationException();
        }
    }

    private static String serializeStringOrXmlDavProperty(DavProperty property) {
        Object value = property.getValue();

        if (value instanceof String) {
            return (String) value;
        } else if (value instanceof ArrayList || value instanceof Element) {
            ListF<Element> elements = (value instanceof Element)
                    ? Cf.list((Element) value)
                    : Cf.x(((ArrayList)value).toArray()).filterByType(Element.class);
            ByteArrayOutputStream os = new ByteArrayOutputStream();
            elements.forEach(e -> DomUtils.I.writeElement(e, os));

            return "<>" + os.toString() + "</>";
        } else {
            return "";
        }
    }

    private enum OperationType {
        REMOVE,
        SET
    }

    @AllArgsConstructor
    private static class Operation {
        final DavPropertyName name;
        final OperationType type;
        final Option<DavProperty> value;

        private String getStringValue() {
            return StringUtils.trim((String) value.get().getValue());
        }

        private String getName() {
            return name.getName();
        }

        public boolean isSet() {
            return type == OperationType.SET;
        }

        public boolean isRemove() {
            return type == OperationType.REMOVE;
        }
    }

    private enum PropertyGroup {
        RESOURCE,
        PUBLIC_URL,
        SHARE_USER,
        SETTING,
        USER_STATE,
        QUIRK_REHASH,
        MS,

        OTHER

    }

    @AllArgsConstructor
    private static class PatchResult {
        private final int status;
        private final String value;

        private PatchResult(int status) {
            this(status, "");
        }

        static PatchResult executeWithOnlyStatus(Function0<MpfsCallbackResponse> requestF) {
            return execute(() -> new PatchResult(requestF.apply().getStatusCode()));
        }

        public static PatchResult execute(Function0<PatchResult> requestF) {
            try {
                return requestF.apply();
            } catch (PermanentHttpFailureException e) {
                return new PatchResult(e.getStatusCode().get(), e.getMessage());
            }
        }
    }

    @AllArgsConstructor
    private static class PatchSemiResult {
        private final Option<DavPropertyName> name;
        private final Option<String> key;
        private final Option<String> value;
        private final Option<Integer> status;
        private final Option<String> namespace;

        public static PatchSemiResult error(DavPropertyName name, int status) {
            return new PatchSemiResult(Option.of(name), Option.empty(), Option.empty(), Option.of(status), Option.empty());
        }

        public static PatchSemiResult set(DavPropertyName name, String key, String value) {
            return new PatchSemiResult(Option.of(name), Option.of(key), Option.of(value), Option.empty(), Option.empty());
        }

        public static PatchSemiResult remove(DavPropertyName name, String key) {
            return new PatchSemiResult(Option.of(name), Option.of(key), Option.empty(), Option.empty(), Option.empty());
        }

        public boolean isError() {
            return name.isPresent() && !value.isPresent() && status.isPresent();
        }

        public boolean isSet() {
            return name.isPresent() && value.isPresent() && !status.isPresent();
        }

        public boolean isRemove() {
            return name.isPresent() && !value.isPresent() && !status.isPresent();
        }

        DavPropertyName getRemoveName() {
            Assert.isTrue(isRemove());
            return name.get();
        }

        String getRemoveKey() {
            Assert.isTrue(isRemove());
            return key.get();
        }

        Tuple2<DavPropertyName, String> getSetNameValue() {
            Assert.isTrue(isSet());
            return Tuple2.tuple(name.get(), value.get());
        }

        Tuple2<String, String> getSetKeyValue() {
            Assert.isTrue(isSet());
            return Tuple2.tuple(key.get(), value.get());
        }

        String getSetKey() {
            Assert.isTrue(isSet());
            return key.get();
        }

        Tuple2<DavPropertyName, Integer> getErrorNameStatus() {
            Assert.isTrue(isError());
            return Tuple2.tuple(name.get(), status.get());
        }

        public int getStatus() {
            Assert.isTrue(isError());
            return status.get();
        }

        PatchSemiResult withNamespace(String namespace) {
            return new PatchSemiResult(name, key, value, status, Option.of(namespace));
        }

        public String getNamespace() {
            return namespace.get();
        }
    }

    private MpfsCallbackResponse consStatusResponse(int statusCode) {
        return new MpfsCallbackResponse(statusCode, "", "", Cf.map(), Option.empty());
    }

    private abstract static class GroupApplier {
        public abstract void apply(
                AuthInfo authInfo, MpfsResource resource, ListF<Operation> operations, MultiStatusResponse response);

        void addToResponse(DavPropertyName name, String value, int status, MultiStatusResponse response) {
            response.add(new DefaultDavProperty<>(name, value), status);
        }

        protected PatchSemiResult operationProcess(AuthInfo authInfo, Operation operation) {
            Option<DavProperties.PropertyDescription> description = DavProperties.find(operation.name);

            boolean preCheck = description.isPresent()
                    && description.get().isAllowWrite()
                    && (operation.isSet() || description.get().isAllowRemove())
                    // CREATIONDATE is the only base-located write-allowed DAV prop
                    && (!description.isSome(DavProperties.CREATIONDATE) || operation.value.isPresent());
            if (!preCheck) {
                return PatchSemiResult.error(operation.name, HttpStatus.SC_403_FORBIDDEN);
            }

            PatchSemiResult semiResult = propertyRecode(authInfo, operation.name, operation.value);

            if (semiResult.isError()
                    || (semiResult.isSet() && operation.isSet())
                    || (semiResult.isRemove() && operation.isRemove()))
            {
                return semiResult;
            } else {
                return PatchSemiResult.error(operation.name, HttpStatus.SC_409_CONFLICT);
            }
        }

        protected PatchSemiResult propertyRecode(
                AuthInfo authInfo, DavPropertyName name, Option<DavProperty> property)
        {
            String key = UrlUtils.urlEncode(DavPropertyNameUtils.getCustomPropertyNameSafe(name));
            Option<String> value = property.map(PropertiesSetter::serializeStringOrXmlDavProperty);
            return new PatchSemiResult(Option.of(name), Option.of(key), value, Option.empty(), Option.empty());
        }
    }

    private abstract class SetPropGroupApplier extends GroupApplier {
        @Override
        public void apply(AuthInfo authInfo, MpfsResource resource,
                ListF<Operation> operations, MultiStatusResponse response)
        {
            ListF<PatchSemiResult> semiResults = operations.map(op -> operationProcess(authInfo, op));

            ListF<PatchSemiResult> set = semiResults.filter(PatchSemiResult::isSet)
                    .stableUniqueBy(PatchSemiResult::getSetKey);
            ListF<PatchSemiResult> remove = semiResults.filter(PatchSemiResult::isRemove).stableUnique();
            ListF<PatchSemiResult> error = semiResults.filter(PatchSemiResult::isError);

            if (set.isNotEmpty() || remove.isNotEmpty()) {
                int status = mpfsClient.setProp(
                        resource.getUser(), resource.getRealPath(),
                        set.map(PatchSemiResult::getSetKeyValue).toTuple2List(Tuple2::get1, Tuple2::get2),
                        remove.map(PatchSemiResult::getRemoveKey)).getStatusCode();

                set.map(PatchSemiResult::getSetNameValue)
                        .forEach(t -> addToResponse(t._1, t._2, status, response));
                remove.map(PatchSemiResult::getRemoveName)
                        .forEach(name -> addToResponse(name, "", status, response));
            }

            error.map(PatchSemiResult::getErrorNameStatus)
                    .forEach(t -> addToResponse(t._1, "", t._2, response));
        }
    }

    private abstract static class SingleApplier extends GroupApplier {
        @Override
        public void apply(
                AuthInfo authInfo, MpfsResource resource, ListF<Operation> operations, MultiStatusResponse response) {
            operations.forEach(op -> {
                PatchResult result = op.isSet() ? set(authInfo, resource, op) : remove(authInfo, resource, op);
                addToResponse(op.name, result.value, result.status, response);
            });
        }

        private PatchResult common(AuthInfo authInfo, Operation operation,
                Function<PatchSemiResult, MpfsCallbackResponse> setF,
                Function<PatchSemiResult, MpfsCallbackResponse> removeF)
        {
            PatchSemiResult semiResult = operationProcess(authInfo, operation);
            if (semiResult.isError()) {
                return new PatchResult(semiResult.getStatus());
            } else if (semiResult.isSet()) {
                return setPostProcess(() -> setF.apply(semiResult));
            } else if (semiResult.isRemove()) {
                return PatchResult.executeWithOnlyStatus(() -> removeF.apply(semiResult));
            } else {
                return new PatchResult(HttpStatus.SC_409_CONFLICT);
            }
        }

        private PatchResult set(AuthInfo authInfo, MpfsResource resource, Operation operation) {
            return getResult(authInfo, resource, operation);
        }

        private PatchResult getResult(AuthInfo authInfo, MpfsResource resource, Operation op) {
            return common(authInfo, op, semiResult -> set(resource, op, semiResult), semiResult -> remove(resource, op, semiResult));
        }

        private PatchResult remove(AuthInfo authInfo, MpfsResource resource, Operation operation) {
            return getResult(authInfo, resource, operation);
        }

        protected PatchResult setPostProcess(Function0<MpfsCallbackResponse> setF) {
            return PatchResult.executeWithOnlyStatus(setF);
        }

        protected abstract MpfsCallbackResponse set(
                MpfsResource resource, Operation operation, PatchSemiResult semiResult);
        protected abstract MpfsCallbackResponse remove(
                MpfsResource resource, Operation operation, PatchSemiResult semiResult);
    }

    private abstract static class SingleApplierWithNamespaceKeyValue extends SingleApplier {
        @Override
        protected MpfsCallbackResponse set(MpfsResource resource, Operation operation, PatchSemiResult semiResult)
        {
            Tuple2<String, String> t = semiResult.getSetKeyValue();
            return set(resource, operation, semiResult.getNamespace(), t._1, t._2);
        }

        @Override
        protected MpfsCallbackResponse remove(MpfsResource resource, Operation operation, PatchSemiResult semiResult) {
            return remove(resource, operation, semiResult.getNamespace(), semiResult.getRemoveKey());
        }

        protected abstract MpfsCallbackResponse set(
                MpfsResource resource, Operation operation, String namespace, String key, String value);
        protected abstract MpfsCallbackResponse remove(
                MpfsResource resource, Operation operation, String namespace, String key);
    }

    private abstract static class SingleApplierWithNoPreCheckRecodeAuth extends SingleApplier {
        @Override
        protected PatchSemiResult operationProcess(AuthInfo authInfo, Operation operation) {
            return operation.isSet()
                    ? PatchSemiResult.set(operation.name, "", "")
                    : PatchSemiResult.remove(operation.name, "");
        }

        @Override
        protected MpfsCallbackResponse set(MpfsResource resource, Operation operation, PatchSemiResult semiResult) {
            return set(resource, operation);
        }

        @Override
        protected MpfsCallbackResponse remove(MpfsResource resource, Operation operation, PatchSemiResult semiResult) {
            return remove(resource, operation);
        }

        protected abstract MpfsCallbackResponse set(MpfsResource resource, Operation operation);
        protected abstract MpfsCallbackResponse remove(MpfsResource resource, Operation operation);

    }
}
