package ru.yandex.chemodan.app.webdav.repository.properties;

import java.util.function.Supplier;

import org.apache.jackrabbit.webdav.property.DavProperty;
import org.apache.jackrabbit.webdav.property.DavPropertyName;
import org.apache.jackrabbit.webdav.property.DefaultDavProperty;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.webdav.auth.AuthInfo;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.chemodan.mpfs.MpfsUserInfo;
import ru.yandex.misc.cache.tl.TlCache;

import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.MAX_FILE_SIZE;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.PAID_MAX_FILE_SIZE;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.PHOTOSTREAM;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.QUOTA_AVAILABLE_BYTES;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.QUOTA_LIMIT_BYTES;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.QUOTA_USED_BYTES;
import static ru.yandex.chemodan.app.webdav.repository.properties.DavProperties.TRASH_USED_BYTES;

/**
 * @author tolmalev
 */
public class UserInfoPropertyFactory implements PropertiesFactory {
    private final MpfsClient mpfsClient;

    private static final Function<MpfsUserInfo, MpfsUserInfo.Space> SPACE = MpfsUserInfo::getSpace;

    public static final MapF<DavPropertyName, OnePropertyFactory<MpfsUserInfo>> properties = Cf.list(
            new OnePropertyFactory<>(QUOTA_AVAILABLE_BYTES, SPACE.andThen(MpfsUserInfo.Space::getFree)),
            new OnePropertyFactory<>(QUOTA_USED_BYTES, SPACE.andThen(MpfsUserInfo.Space::getUsed)),

            new OnePropertyFactory<>(QUOTA_LIMIT_BYTES, SPACE.andThen(MpfsUserInfo.Space::getLimit)),
            new OnePropertyFactory<>(MAX_FILE_SIZE, SPACE.andThen(MpfsUserInfo.Space::getFilesize_limit)),
            new OnePropertyFactory<>(PAID_MAX_FILE_SIZE, SPACE.andThen(MpfsUserInfo.Space::getPaid_filesize_limit)),
            new OnePropertyFactory<>(TRASH_USED_BYTES, SPACE.andThen(MpfsUserInfo.Space::getTrash)),

            new OnePropertyFactory<MpfsUserInfo>(PHOTOSTREAM, (info, authInfo) ->
                    authInfo.ourClient
                            .filter(ourClient -> ourClient.installId.isPresent())
                            .flatMapO(ourClient -> info.settings.photostream.getO(ourClient.installId.get()))
                            .map(XmlPropertyUtils::hackYadropXml)
            )
    ).toMapMappingToKey(OnePropertyFactory::getName);

    public UserInfoPropertyFactory(MpfsClient mpfsClient) {
        this.mpfsClient = mpfsClient;
    }

    @Override
    public boolean accepts(DavPropertyName name) {
        return name.equals(DavProperties.STATE_ALLPROP.name) ||
                properties.containsKeyTs(name);
    }

    @Override
    public ListF<DavProperty> cons(Supplier<MpfsFileInfo> info, AuthInfo auth, DavPropertyName name) {
        //XXX: looks bad
        if (name.equals(DavProperties.STATE_ALLPROP.name)) {
            MpfsUserInfo userInfo = getUserInfo(auth);
            return userInfo.states.getO("global").flatMap(states -> states.entries().map(t2 ->
                    new DefaultDavProperty<>(t2._1, t2._2, DavProperties.USER_STATES)
            ));
        }
        return properties.getO(name).flatMapO(p -> p.cons(getUserInfo(auth), auth));
    }

    private MpfsUserInfo getUserInfo(AuthInfo uid) {
        return TlCache.getOrElseUpdate("mpfs-user-info-" + uid.getUidStr(), () -> mpfsClient.getUserInfoObj(uid));
    }

    @Override
    public int order() {
        return 1;
    }
}
