package ru.yandex.chemodan.app.webdav.repository.upload;

import org.apache.http.client.utils.DateUtils;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.util.http.RequestUtils;
import ru.yandex.commune.uploader.util.http.HttpContentRange;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.regex.Pattern2;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @author tolmalev
 */
public class UploadUtils {
    private static final Pattern2 typePattern = Pattern2.compile("^[a-zA-Z0-9]*$");

    public static Upload extractUploadFromRequest(HttpServletRequestX reqX) {
        Upload upload = new Upload();

        upload.yandexDiff = getYandexDiff(reqX);
        upload.replaceMd5 = RequestUtils.getIfMatch(reqX);

        if (upload.yandexDiff.isPresent()) {
            upload.md5 = upload.yandexDiff;
            upload.mode = Upload.Mode.DELTA;
        } else {
            upload.md5 = reqX.getHeaderO("X-Yandex-Target-MD5").orElse(reqX.getHeaderO("ETag"));
        }

        upload.sha256 = reqX.getHeaderO("Sha256");
        upload.contentLength = reqX.getHeaderO("Content-Length").flatMapO(Cf.Long::parseSafe);
        upload.contentEncoding = reqX.getHeaderO("Content-Encoding");
        upload.contentType = reqX.getHeaderO("Content-Type");

        // iOS live-photo: CHEMODAN-40827
        upload.livePhotoMd5 = reqX.getHeaderO("X-Yandex-Live-Photo-Md5");
        upload.livePhotoSha256 = reqX.getHeaderO("X-Yandex-Live-Photo-Sha256");
        upload.livePhotoType = reqX.getHeaderO("X-Yandex-Live-Photo-Type");
        upload.livePhotoSize = reqX.getHeaderO("X-Yandex-Live-Photo-Size").filterMap(Cf.Long::parseSafe);
        upload.livePhotoOperation = reqX.getHeaderO("X-Yandex-Live-Photo-Operation");

        upload.photostreamDestination = reqX.getHeaderO("X-Yandex-Photostream-Destination");

        upload.deviceResourceSubtype = reqX.getHeaderO("X-Yandex-Device-Resource-Subtype");
        upload.deviceCollections = reqX.getHeaderO("X-Yandex-Device-Collections").map(UrlUtils::urlDecode);
        upload.deviceOriginalPath = reqX.getHeaderO("X-Yandex-Device-Original-Path").map(UrlUtils::urlDecode);

        upload.sourceId = reqX.getHeaderO("Yandex-Disk-Client-File-Id");
        upload.forceDeletionLogDeduplication = reqX.getHeaderO("X-Yandex-Force-Deletion-Log-Deduplication");

        upload.remoteIpAdress = reqX.getRemoteIpAddress();

        //https://github.yandex-team.ru/erlang/yadrop/blob/master/apps/yadrop/src/yadrop_stream.erl#L272
        upload.force = !reqX.getHeaderO("If-None-Match").isSome("*");
        upload.hidden = parseHidden(reqX);

        parseDisposition(reqX, upload);

        try {
            upload.contentRange = reqX.getHeaderO("Content-Range").map(HttpContentRange.parseF);
        } catch (RuntimeException ignored) {}

        upload.size = upload.contentRange.map(r -> r.instanceLength).orElse(upload.contentLength);

        if (upload.mode == Upload.Mode.NORMAL && upload.contentRange.filterNot(HttpContentRange::isFullEntity).isPresent()) {
            upload.mode = Upload.Mode.RESUME;
        }

        return upload;
    }

    private static boolean parseHidden(HttpServletRequestX reqX) {
        //TODO: autohide
        return reqX.getHeaderO("X-Yandex-Hidden").map("T"::equals)
                .getOrElse(false);
    }

    private static void parseDisposition(HttpServletRequestX reqX, Upload upload) {
        reqX.getHeaderO("Content-Disposition").forEach(disp -> parseDisposition(disp, upload));
    }

    static void parseDisposition(String disp, Upload upload) {
        ListF<String> parts = Cf.list(disp.split(";")).map(StringUtils::strip);

        if (parts.isNotEmpty()) {
            parseType(parts.first(), upload);
        }
        upload.isScreenshot = upload.type.isSome("screenshot");

        parseDispositionKv(parts.drop(1), upload);
    }

    private static void parseType(String type, Upload upload) {
        if ("".equals(type) || "attachment".equals(type) || "inline".equals(type)) {
            return;
        }
        if (typePattern.matches(type)) {
            upload.type = Option.of(type);
        }
    }

    private static void parseDispositionKv(ListF<String> parts, Upload upload) {
        parts.forEach(part -> {
            if (part.startsWith("creation-date=")) {
                upload.ctime = parseDate(StringUtils.removeStart(part, "creation-date="));
            } else if (part.startsWith("modification-date=")) {
                upload.mtime = parseDate(StringUtils.removeStart(part, "modification-date="));
            } else if (part.startsWith("e-date=")) {
                upload.etime = parseDate(StringUtils.removeStart(part, "e-date="));
            } else if (part.equals("public=true")) {
                upload.isPublic = true;
            }
        });
    }

    static Option<Instant> parseDate(String tag) {
        tag = StringUtils.strip(tag, "\" ");
        try {
            return Option.of(new Instant(DateUtils.parseDate(tag)));
        } catch (RuntimeException e) {
            return Option.empty();
        }
    }

    private static Option<String> getYandexDiff(HttpServletRequestX reqX) {
        //todo: maybe add validation of hex
        return reqX.getHeaderO("Yandex-Diff").map(diff -> {
            if (diff.startsWith("\"")) {
                return StringUtils.strip(diff, "\"");
            }
            return diff;
        });
    }
}
